/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.dfs;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.dfs.Block;
import org.apache.hadoop.dfs.BlockMetaDataInfo;
import org.apache.hadoop.dfs.DataNode;
import org.apache.hadoop.dfs.DatanodeID;
import org.apache.hadoop.dfs.DatanodeProtocol;
import org.apache.hadoop.dfs.FSNamesystem;
import org.apache.hadoop.dfs.INodeFileUnderConstruction;
import org.apache.hadoop.dfs.InterDatanodeProtocol;
import org.apache.hadoop.dfs.StringBytesWritable;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
class LeaseManager {
    static final Log LOG = LogFactory.getLog(LeaseManager.class);
    private final FSNamesystem fsnamesystem;
    private long softLimit = 60000L;
    private long hardLimit = 3600000L;
    private SortedMap<StringBytesWritable, Lease> leases = new TreeMap<StringBytesWritable, Lease>();
    private SortedSet<Lease> sortedLeases = new TreeSet<Lease>();
    private SortedMap<String, Lease> sortedLeasesByPath = new TreeMap<String, Lease>();
    private static Map<Block, Block> ongoingRecovery = new HashMap<Block, Block>();

    LeaseManager(FSNamesystem fsnamesystem) {
        this.fsnamesystem = fsnamesystem;
    }

    Lease getLease(StringBytesWritable holder) throws IOException {
        return (Lease)this.leases.get(holder);
    }

    SortedSet<Lease> getSortedLeases() {
        return this.sortedLeases;
    }

    Lease getLeaseByPath(String src) {
        return (Lease)this.sortedLeasesByPath.get(src);
    }

    synchronized int countLease() {
        return this.sortedLeases.size();
    }

    synchronized int countPath() {
        int count = 0;
        for (Lease lease : this.sortedLeases) {
            count += lease.getPaths().size();
        }
        return count;
    }

    synchronized void addLease(StringBytesWritable holder, String src) throws IOException {
        Lease lease = this.getLease(holder);
        if (lease == null) {
            lease = new Lease(holder);
            this.leases.put(holder, lease);
            this.sortedLeases.add(lease);
        } else {
            this.renewLease(lease);
        }
        this.sortedLeasesByPath.put(src, lease);
        lease.paths.add(new StringBytesWritable(src));
    }

    synchronized void removeLease(Lease lease, String src) throws IOException {
        this.sortedLeasesByPath.remove(src);
        if (!lease.paths.remove(new StringBytesWritable(src))) {
            LOG.error((Object)(src + " not found in lease.paths (=" + lease.paths + ")"));
        }
        if (!lease.hasPath()) {
            this.leases.remove(lease.holder);
            if (!this.sortedLeases.remove(lease)) {
                LOG.error((Object)(lease + " not found in sortedLeases"));
            }
        }
    }

    synchronized void removeLease(StringBytesWritable holder, String src) throws IOException {
        this.removeLease(this.getLease(holder), src);
    }

    synchronized String findPath(INodeFileUnderConstruction pendingFile) throws IOException {
        String src;
        Lease lease = this.getLease(pendingFile.clientName);
        if (lease != null && (src = lease.findPath(pendingFile)) != null) {
            return src;
        }
        throw new IOException("pendingFile (=" + pendingFile + ") not found." + "(lease=" + lease + ")");
    }

    synchronized void renewLease(String holder) throws IOException {
        this.renewLease(this.getLease(new StringBytesWritable(holder)));
    }

    synchronized void renewLease(Lease lease) {
        if (lease != null) {
            this.sortedLeases.remove(lease);
            lease.renew();
            this.sortedLeases.add(lease);
        }
    }

    synchronized void changeLease(String src, String dst, String overwrite, String replaceBy) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)(this.getClass().getName() + ".changelease: " + " src=" + src + ", dest=" + dst + ", overwrite=" + overwrite + ", replaceBy=" + replaceBy));
        }
        TreeMap<String, Lease> addTo = new TreeMap<String, Lease>();
        SortedMap<String, Lease> myset = this.sortedLeasesByPath.tailMap(src);
        int srclen = src.length();
        Iterator<Map.Entry<String, Lease>> iter = myset.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<String, Lease> entry = iter.next();
            String path = entry.getKey();
            if (!path.startsWith(src)) {
                if (!LOG.isDebugEnabled()) break;
                LOG.debug((Object)("changelease comparing " + path + " with " + src + " and terminating."));
                break;
            }
            if (path.length() > srclen && path.charAt(srclen) != '/') continue;
            Lease lease = entry.getValue();
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("changelease comparing " + path + " with " + src + " and replacing "));
            }
            lease.replacePrefix(src, overwrite, replaceBy);
            String newPath = path.replaceFirst(overwrite, replaceBy);
            addTo.put(newPath, lease);
            iter.remove();
        }
        this.sortedLeasesByPath.putAll(addTo);
    }

    void setLeasePeriod(long softLimit, long hardLimit) {
        this.softLimit = softLimit;
        this.hardLimit = hardLimit;
    }

    Monitor createMonitor() {
        return new Monitor();
    }

    static void recoverBlocks(Block[] blocks, DatanodeID[][] targets, DatanodeProtocol namenode, Configuration conf) {
        for (int i = 0; i < blocks.length; ++i) {
            try {
                LeaseManager.recoverBlock(blocks[i], targets[i], namenode, conf, true);
                continue;
            }
            catch (IOException e) {
                LOG.warn((Object)("recoverBlocks, i=" + i), (Throwable)e);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static Block recoverBlock(Block block, DatanodeID[] datanodeids, DatanodeProtocol namenode, Configuration conf, boolean closeFile) throws IOException {
        Map<Block, Block> map = ongoingRecovery;
        synchronized (map) {
            Block tmp = new Block();
            tmp.set(block.blkid, block.len, 1L);
            if (ongoingRecovery.get(tmp) != null) {
                String msg = "Block " + block + " is already being recovered, " + " ignoring this request to recover it.";
                LOG.info((Object)msg);
                throw new IOException(msg);
            }
            ongoingRecovery.put(block, block);
        }
        try {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("block=" + block + ", datanodeids=" + Arrays.asList(datanodeids)));
            }
            ArrayList<BlockRecord> syncList = new ArrayList<BlockRecord>();
            long minlength = Long.MAX_VALUE;
            int errorCount = 0;
            for (DatanodeID id : datanodeids) {
                try {
                    InterDatanodeProtocol datanode = DataNode.createInterDataNodeProtocolProxy(id, conf);
                    BlockMetaDataInfo info = datanode.getBlockMetaDataInfo(block);
                    if (info == null || info.getGenerationStamp() < block.generationStamp) continue;
                    syncList.add(new BlockRecord(id, datanode, new Block(info)));
                    if (info.len >= minlength) continue;
                    minlength = info.len;
                }
                catch (IOException e) {
                    ++errorCount;
                    InterDatanodeProtocol.LOG.warn((Object)("Failed to getBlockMetaDataInfo for block (=" + block + ") from datanode (=" + id + ")"), (Throwable)e);
                }
            }
            if (syncList.isEmpty() && errorCount > 0) {
                throw new IOException("All datanodes failed: block=" + block + ", datanodeids=" + Arrays.asList(datanodeids));
            }
            Block block2 = LeaseManager.syncBlock(block, minlength, syncList, namenode, closeFile);
            return block2;
        }
        finally {
            Map<Block, Block> map2 = ongoingRecovery;
            synchronized (map2) {
                ongoingRecovery.remove(block);
            }
        }
    }

    private static Block syncBlock(Block block, long minlength, List<BlockRecord> syncList, DatanodeProtocol namenode, boolean closeFile) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("block=" + block + ", minlength=" + minlength + ", syncList=" + syncList + ", closeFile=" + closeFile));
        }
        if (syncList.isEmpty()) {
            namenode.commitBlockSynchronization(block, 0L, 0L, closeFile, true, DatanodeID.EMPTY_ARRAY);
            return null;
        }
        ArrayList<DatanodeID> successList = new ArrayList<DatanodeID>();
        long generationstamp = namenode.nextGenerationStamp(block);
        Block newblock = new Block(block.blkid, minlength, generationstamp);
        for (BlockRecord r : syncList) {
            try {
                r.datanode.updateBlock(r.block, newblock, closeFile);
                successList.add(r.id);
            }
            catch (IOException e) {
                InterDatanodeProtocol.LOG.warn((Object)("Failed to updateBlock (newblock=" + newblock + ", datanode=" + r.id + ")"), (Throwable)e);
            }
        }
        if (!successList.isEmpty()) {
            namenode.commitBlockSynchronization(block, newblock.generationStamp, newblock.len, closeFile, false, successList.toArray(new DatanodeID[successList.size()]));
            return newblock;
        }
        return null;
    }

    private static class BlockRecord {
        final DatanodeID id;
        final InterDatanodeProtocol datanode;
        final Block block;

        BlockRecord(DatanodeID id, InterDatanodeProtocol datanode, Block block) {
            this.id = id;
            this.datanode = datanode;
            this.block = block;
        }

        public String toString() {
            return "block:" + this.block + " node:" + this.id;
        }
    }

    class Monitor
    implements Runnable {
        Monitor() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void run() {
            try {
                while (((LeaseManager)LeaseManager.this).fsnamesystem.fsRunning) {
                    FSNamesystem fSNamesystem = LeaseManager.this.fsnamesystem;
                    synchronized (fSNamesystem) {
                        LeaseManager leaseManager = LeaseManager.this;
                        synchronized (leaseManager) {
                            Lease top;
                            while (LeaseManager.this.sortedLeases.size() > 0 && (top = (Lease)LeaseManager.this.sortedLeases.first()) != null && top.expiredHardLimit()) {
                                LOG.info((Object)("Lease Monitor: Removing lease " + top + ", sortedLeases.size()=: " + LeaseManager.this.sortedLeases.size()));
                                for (StringBytesWritable s : top.paths) {
                                    LeaseManager.this.fsnamesystem.internalReleaseLease(top, s.getString());
                                }
                            }
                        }
                    }
                    try {
                        Thread.sleep(2000L);
                    }
                    catch (InterruptedException ie) {
                        if (!LOG.isDebugEnabled()) continue;
                        LOG.debug((Object)(this.getClass().getName() + " is interrupted"), (Throwable)ie);
                    }
                }
            }
            catch (Exception e) {
                LOG.error((Object)("In " + this.getClass().getName()), (Throwable)e);
            }
        }
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    class Lease
    implements Comparable<Lease> {
        private StringBytesWritable holder;
        private long lastUpdate;
        private Collection<StringBytesWritable> paths = new TreeSet<StringBytesWritable>();

        private Lease(StringBytesWritable holder) throws IOException {
            this.holder = holder;
            this.renew();
        }

        private void renew() {
            this.lastUpdate = FSNamesystem.now();
        }

        public boolean expiredHardLimit() {
            return FSNamesystem.now() - this.lastUpdate > LeaseManager.this.hardLimit;
        }

        public boolean expiredSoftLimit() {
            return FSNamesystem.now() - this.lastUpdate > LeaseManager.this.softLimit;
        }

        private String findPath(INodeFileUnderConstruction pendingFile) {
            Iterator<StringBytesWritable> i = this.paths.iterator();
            while (i.hasNext()) {
                String src = i.next().toString();
                if (((LeaseManager)LeaseManager.this).fsnamesystem.dir.getFileINode(src) != pendingFile) continue;
                return src;
            }
            return null;
        }

        boolean hasPath() {
            return !this.paths.isEmpty();
        }

        public String toString() {
            return "[Lease.  Holder: " + this.holder + ", pendingcreates: " + this.paths.size() + "]";
        }

        @Override
        public int compareTo(Lease o) {
            Lease l1 = this;
            Lease l2 = o;
            long lu1 = l1.lastUpdate;
            long lu2 = l2.lastUpdate;
            if (lu1 < lu2) {
                return -1;
            }
            if (lu1 > lu2) {
                return 1;
            }
            return l1.holder.compareTo(l2.holder);
        }

        public boolean equals(Object o) {
            if (!(o instanceof Lease)) {
                return false;
            }
            Lease obj = (Lease)o;
            return this.lastUpdate == obj.lastUpdate && this.holder.equals(obj.holder);
        }

        public int hashCode() {
            return this.holder.hashCode();
        }

        Collection<StringBytesWritable> getPaths() {
            return this.paths;
        }

        void replacePrefix(String src, String overwrite, String replaceBy) throws IOException {
            ArrayList<StringBytesWritable> toAdd = new ArrayList<StringBytesWritable>();
            Iterator<StringBytesWritable> f = this.paths.iterator();
            while (f.hasNext()) {
                String path = f.next().getString();
                if (!path.startsWith(src)) continue;
                f.remove();
                String newPath = path.replaceFirst(overwrite, replaceBy);
                toAdd.add(new StringBytesWritable(newPath));
                LOG.debug((Object)("Modified Lease for file " + path + " to new path " + newPath));
            }
            f = toAdd.iterator();
            while (f.hasNext()) {
                this.paths.add(f.next());
            }
        }
    }
}

