/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Arrays;
import java.util.zip.CRC32;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.ChecksumException;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FSInputChecker;
import org.apache.hadoop.fs.FSOutputSummer;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.FilterFileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.util.Progressable;
import org.apache.hadoop.util.StringUtils;

public abstract class ChecksumFileSystem
extends FilterFileSystem {
    private static final byte[] CHECKSUM_VERSION = new byte[]{99, 114, 99, 0};
    private int bytesPerChecksum = 512;
    private static final PathFilter DEFAULT_FILTER = new PathFilter(){

        public boolean accept(Path file) {
            return !ChecksumFileSystem.isChecksumFile(file);
        }
    };

    public static double getApproxChkSumLength(long size) {
        return 0.01f * (float)size;
    }

    public ChecksumFileSystem(FileSystem fs) {
        super(fs);
    }

    public void setConf(Configuration conf) {
        super.setConf(conf);
        if (conf != null) {
            this.bytesPerChecksum = conf.getInt("io.bytes.per.checksum", 512);
        }
    }

    public FileSystem getRawFileSystem() {
        return this.fs;
    }

    public Path getChecksumFile(Path file) {
        return new Path(file.getParent(), "." + file.getName() + ".crc");
    }

    public static boolean isChecksumFile(Path file) {
        String name = file.getName();
        return name.startsWith(".") && name.endsWith(".crc");
    }

    public long getChecksumFileLength(Path file, long fileSize) {
        return ChecksumFileSystem.getChecksumLength(fileSize, this.getBytesPerSum());
    }

    public int getBytesPerSum() {
        return this.bytesPerChecksum;
    }

    private int getSumBufferSize(int bytesPerSum, int bufferSize) {
        int defaultBufferSize = this.getConf().getInt("io.file.buffer.size", 4096);
        int proportionalBufferSize = bufferSize / bytesPerSum;
        return Math.max(bytesPerSum, Math.max(proportionalBufferSize, defaultBufferSize));
    }

    public FSDataInputStream open(Path f, int bufferSize) throws IOException {
        return new FSDataInputStream(new ChecksumFSInputChecker(this, f, bufferSize));
    }

    public static long getChecksumLength(long size, int bytesPerSum) {
        return (size + (long)bytesPerSum - 1L) / (long)bytesPerSum * 4L + (long)CHECKSUM_VERSION.length + 4L;
    }

    public FSDataOutputStream create(Path f, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        Path parent = f.getParent();
        if (parent != null && !this.mkdirs(parent)) {
            throw new IOException("Mkdirs failed to create " + parent);
        }
        return new FSDataOutputStream(new ChecksumFSOutputSummer(this, f, overwrite, bufferSize, replication, blockSize, progress), null);
    }

    public boolean setReplication(Path src, short replication) throws IOException {
        boolean value = this.fs.setReplication(src, replication);
        if (!value) {
            return false;
        }
        Path checkFile = this.getChecksumFile(src);
        if (this.exists(checkFile)) {
            this.fs.setReplication(checkFile, replication);
        }
        return true;
    }

    public boolean rename(Path src, Path dst) throws IOException {
        if (this.fs.isDirectory(src)) {
            return this.fs.rename(src, dst);
        }
        boolean value = this.fs.rename(src, dst);
        if (!value) {
            return false;
        }
        Path checkFile = this.getChecksumFile(src);
        if (this.fs.exists(checkFile)) {
            value = this.fs.isDirectory(dst) ? this.fs.rename(checkFile, dst) : this.fs.rename(checkFile, this.getChecksumFile(dst));
        }
        return value;
    }

    public boolean delete(Path f, boolean recursive) throws IOException {
        FileStatus fstatus = null;
        try {
            fstatus = this.fs.getFileStatus(f);
        }
        catch (FileNotFoundException e) {
            return false;
        }
        if (fstatus.isDir()) {
            return this.fs.delete(f, recursive);
        }
        Path checkFile = this.getChecksumFile(f);
        if (this.fs.exists(checkFile)) {
            this.fs.delete(checkFile, true);
        }
        return this.fs.delete(f, true);
    }

    public FileStatus[] listStatus(Path f) throws IOException {
        return this.fs.listStatus(f, DEFAULT_FILTER);
    }

    public boolean mkdirs(Path f) throws IOException {
        return this.fs.mkdirs(f);
    }

    public void copyFromLocalFile(boolean delSrc, Path src, Path dst) throws IOException {
        Configuration conf = this.getConf();
        FileUtil.copy(ChecksumFileSystem.getLocal(conf), src, this, dst, delSrc, conf);
    }

    public void copyToLocalFile(boolean delSrc, Path src, Path dst) throws IOException {
        Configuration conf = this.getConf();
        FileUtil.copy(this, src, ChecksumFileSystem.getLocal(conf), dst, delSrc, conf);
    }

    public void copyToLocalFile(Path src, Path dst, boolean copyCrc) throws IOException {
        if (!this.fs.isDirectory(src)) {
            this.fs.copyToLocalFile(src, dst);
            FileSystem localFs = ChecksumFileSystem.getLocal(this.getConf());
            if (localFs instanceof ChecksumFileSystem) {
                localFs = ((ChecksumFileSystem)localFs).getRawFileSystem();
            }
            if (localFs.isDirectory(dst)) {
                dst = new Path(dst, src.getName());
            }
            if (localFs.exists(dst = this.getChecksumFile(dst))) {
                localFs.delete(dst, true);
            }
            Path checksumFile = this.getChecksumFile(src);
            if (copyCrc && this.fs.exists(checksumFile)) {
                this.fs.copyToLocalFile(checksumFile, dst);
            }
        } else {
            FileStatus[] srcs;
            for (FileStatus srcFile : srcs = this.listStatus(src)) {
                this.copyToLocalFile(srcFile.getPath(), new Path(dst, srcFile.getPath().getName()), copyCrc);
            }
        }
    }

    public Path startLocalOutput(Path fsOutputFile, Path tmpLocalFile) throws IOException {
        return tmpLocalFile;
    }

    public void completeLocalOutput(Path fsOutputFile, Path tmpLocalFile) throws IOException {
        this.moveFromLocalFile(tmpLocalFile, fsOutputFile);
    }

    public boolean reportChecksumFailure(Path f, FSDataInputStream in, long inPos, FSDataInputStream sums, long sumsPos) {
        return false;
    }

    private static class ChecksumFSOutputSummer
    extends FSOutputSummer {
        private FSDataOutputStream datas;
        private FSDataOutputStream sums;
        private static final float CHKSUM_AS_FRACTION = 0.01f;

        public ChecksumFSOutputSummer(ChecksumFileSystem fs, Path file, boolean overwrite, short replication, long blockSize, Configuration conf) throws IOException {
            this(fs, file, overwrite, conf.getInt("io.file.buffer.size", 4096), replication, blockSize, null);
        }

        public ChecksumFSOutputSummer(ChecksumFileSystem fs, Path file, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
            super(new CRC32(), fs.getBytesPerSum(), 4);
            int bytesPerSum = fs.getBytesPerSum();
            this.datas = fs.getRawFileSystem().create(file, overwrite, bufferSize, replication, blockSize, progress);
            int sumBufferSize = fs.getSumBufferSize(bytesPerSum, bufferSize);
            this.sums = fs.getRawFileSystem().create(fs.getChecksumFile(file), true, sumBufferSize, replication, blockSize);
            this.sums.write(CHECKSUM_VERSION, 0, CHECKSUM_VERSION.length);
            this.sums.writeInt(bytesPerSum);
        }

        public void close() throws IOException {
            this.flushBuffer();
            this.sums.close();
            this.datas.close();
        }

        protected void writeChunk(byte[] b, int offset, int len, byte[] checksum) throws IOException {
            this.datas.write(b, offset, len);
            this.sums.write(checksum);
        }
    }

    private static class ChecksumFSInputChecker
    extends FSInputChecker {
        public static final Log LOG = LogFactory.getLog((String)"org.apache.hadoop.fs.FSInputChecker");
        private ChecksumFileSystem fs;
        private FSDataInputStream datas;
        private FSDataInputStream sums;
        private static final int HEADER_LENGTH = 8;
        private int bytesPerSum = 1;
        private long fileLen = -1L;

        public ChecksumFSInputChecker(ChecksumFileSystem fs, Path file) throws IOException {
            this(fs, file, fs.getConf().getInt("io.file.buffer.size", 4096));
        }

        public ChecksumFSInputChecker(ChecksumFileSystem fs, Path file, int bufferSize) throws IOException {
            super(file, fs.getFileStatus(file).getReplication());
            this.datas = fs.getRawFileSystem().open(file, bufferSize);
            this.fs = fs;
            Path sumFile = fs.getChecksumFile(file);
            try {
                int sumBufferSize = fs.getSumBufferSize(fs.getBytesPerSum(), bufferSize);
                this.sums = fs.getRawFileSystem().open(sumFile, sumBufferSize);
                byte[] version = new byte[CHECKSUM_VERSION.length];
                this.sums.readFully(version);
                if (!Arrays.equals(version, CHECKSUM_VERSION)) {
                    throw new IOException("Not a checksum file: " + sumFile);
                }
                this.bytesPerSum = this.sums.readInt();
                this.set(new CRC32(), this.bytesPerSum, 4);
            }
            catch (FileNotFoundException e) {
                this.set(null, 1, 0);
            }
            catch (IOException e) {
                LOG.warn((Object)("Problem opening checksum file: " + file + ".  Ignoring exception: " + StringUtils.stringifyException(e)));
                this.set(null, 1, 0);
            }
        }

        private long getChecksumFilePos(long dataPos) {
            return 8L + 4L * (dataPos / (long)this.bytesPerSum);
        }

        protected long getChunkPosition(long dataPos) {
            return dataPos / (long)this.bytesPerSum * (long)this.bytesPerSum;
        }

        public int available() throws IOException {
            return this.datas.available() + super.available();
        }

        public int read(long position, byte[] b, int off, int len) throws IOException {
            if ((off | len | off + len | b.length - (off + len)) < 0) {
                throw new IndexOutOfBoundsException();
            }
            if (len == 0) {
                return 0;
            }
            if (position < 0L) {
                throw new IllegalArgumentException("Parameter position can not to be negative");
            }
            ChecksumFSInputChecker checker = new ChecksumFSInputChecker(this.fs, this.file);
            checker.seek(position);
            int nread = checker.read(b, off, len);
            checker.close();
            return nread;
        }

        public void close() throws IOException {
            this.datas.close();
            if (this.sums != null) {
                this.sums.close();
            }
            this.set(null, 1, 0);
        }

        public boolean seekToNewSource(long targetPos) throws IOException {
            long sumsPos = this.getChecksumFilePos(targetPos);
            this.fs.reportChecksumFailure(this.file, this.datas, targetPos, this.sums, sumsPos);
            boolean newDataSource = this.datas.seekToNewSource(targetPos);
            return this.sums.seekToNewSource(sumsPos) || newDataSource;
        }

        protected int readChunk(long pos, byte[] buf, int offset, int len, byte[] checksum) throws IOException {
            boolean eof = false;
            if (this.needChecksum()) {
                try {
                    long checksumPos = this.getChecksumFilePos(pos);
                    if (checksumPos != this.sums.getPos()) {
                        this.sums.seek(checksumPos);
                    }
                    this.sums.readFully(checksum);
                }
                catch (EOFException e) {
                    eof = true;
                }
                len = this.bytesPerSum;
            }
            if (pos != this.datas.getPos()) {
                this.datas.seek(pos);
            }
            int nread = ChecksumFSInputChecker.readFully(this.datas, buf, offset, len);
            if (eof && nread > 0) {
                throw new ChecksumException("Checksum error: " + this.file + " at " + pos, pos);
            }
            return nread;
        }

        private long getFileLength() throws IOException {
            if (this.fileLen == -1L) {
                this.fileLen = this.fs.getContentSummary(this.file).getLength();
            }
            return this.fileLen;
        }

        public synchronized long skip(long n) throws IOException {
            long fileLength;
            long curPos = this.getPos();
            if (n + curPos > (fileLength = this.getFileLength())) {
                n = fileLength - curPos;
            }
            return super.skip(n);
        }

        public synchronized void seek(long pos) throws IOException {
            if (pos > this.getFileLength()) {
                throw new IOException("Cannot seek after EOF");
            }
            super.seek(pos);
        }
    }
}

