/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.mapred;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.mapred.DefaultJobHistoryParser;
import org.apache.hadoop.mapred.JobHistory;
import org.apache.hadoop.util.StringUtils;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
class HistoryViewer {
    private static SimpleDateFormat dateFormat = new SimpleDateFormat("d-MMM-yyyy HH:mm:ss");
    private FileSystem fs;
    private Configuration conf;
    private Path historyLogDir;
    private String jobLogFile;
    private JobHistory.JobInfo job;
    private String trackerHostName;
    private String trackerStartTime;
    private String jobId;
    private boolean printAll;
    private PathFilter jobLogFileFilter = new PathFilter(){

        public boolean accept(Path path) {
            return !path.getName().endsWith(".xml");
        }
    };
    private Comparator<JobHistory.Task> cMap = new Comparator<JobHistory.Task>(){

        @Override
        public int compare(JobHistory.Task t1, JobHistory.Task t2) {
            long l1 = t1.getLong(JobHistory.Keys.FINISH_TIME) - t1.getLong(JobHistory.Keys.START_TIME);
            long l2 = t2.getLong(JobHistory.Keys.FINISH_TIME) - t2.getLong(JobHistory.Keys.START_TIME);
            return l2 < l1 ? -1 : (l2 == l1 ? 0 : 1);
        }
    };
    private Comparator<JobHistory.Task> cShuffle = new Comparator<JobHistory.Task>(){

        @Override
        public int compare(JobHistory.Task t1, JobHistory.Task t2) {
            long l1 = t1.getLong(JobHistory.Keys.SHUFFLE_FINISHED) - t1.getLong(JobHistory.Keys.START_TIME);
            long l2 = t2.getLong(JobHistory.Keys.SHUFFLE_FINISHED) - t2.getLong(JobHistory.Keys.START_TIME);
            return l2 < l1 ? -1 : (l2 == l1 ? 0 : 1);
        }
    };
    private Comparator<JobHistory.Task> cFinishShuffle = new Comparator<JobHistory.Task>(){

        @Override
        public int compare(JobHistory.Task t1, JobHistory.Task t2) {
            long l1 = t1.getLong(JobHistory.Keys.SHUFFLE_FINISHED);
            long l2 = t2.getLong(JobHistory.Keys.SHUFFLE_FINISHED);
            return l2 < l1 ? -1 : (l2 == l1 ? 0 : 1);
        }
    };
    private Comparator<JobHistory.Task> cFinishMapRed = new Comparator<JobHistory.Task>(){

        @Override
        public int compare(JobHistory.Task t1, JobHistory.Task t2) {
            long l1 = t1.getLong(JobHistory.Keys.FINISH_TIME);
            long l2 = t2.getLong(JobHistory.Keys.FINISH_TIME);
            return l2 < l1 ? -1 : (l2 == l1 ? 0 : 1);
        }
    };
    private Comparator<JobHistory.Task> cReduce = new Comparator<JobHistory.Task>(){

        @Override
        public int compare(JobHistory.Task t1, JobHistory.Task t2) {
            long l1 = t1.getLong(JobHistory.Keys.FINISH_TIME) - t1.getLong(JobHistory.Keys.SHUFFLE_FINISHED);
            long l2 = t2.getLong(JobHistory.Keys.FINISH_TIME) - t2.getLong(JobHistory.Keys.SHUFFLE_FINISHED);
            return l2 < l1 ? -1 : (l2 == l1 ? 0 : 1);
        }
    };

    public HistoryViewer(String outputDir, Configuration conf, boolean printAll) throws IOException {
        this.conf = conf;
        this.printAll = printAll;
        Path output = new Path(outputDir);
        this.historyLogDir = new Path(output, "_logs/history");
        try {
            this.fs = output.getFileSystem(this.conf);
            if (!this.fs.exists(output)) {
                throw new IOException("History directory " + this.historyLogDir.toString() + "does not exist");
            }
            Path[] jobFiles = FileUtil.stat2Paths(this.fs.listStatus(this.historyLogDir, this.jobLogFileFilter));
            if (jobFiles.length == 0) {
                throw new IOException("Not a valid history directory " + this.historyLogDir.toString());
            }
            this.jobLogFile = jobFiles[0].toString();
            String[] jobDetails = JobHistory.JobInfo.decodeJobHistoryFileName(jobFiles[0].getName()).split("_");
            this.trackerHostName = jobDetails[0];
            this.trackerStartTime = jobDetails[1];
            this.jobId = jobDetails[2] + "_" + jobDetails[3] + "_" + jobDetails[4];
            this.job = new JobHistory.JobInfo(this.jobId);
            DefaultJobHistoryParser.parseJobTasks(jobFiles[0].toString(), this.job, this.fs);
        }
        catch (Exception e) {
            throw new IOException("Not able to initialize History viewer");
        }
    }

    public void print() throws IOException {
        this.printJobDetails();
        this.printTaskSummary();
        this.printJobAnalysis();
        this.printTasks("MAP", "FAILED");
        this.printTasks("MAP", "KILLED");
        this.printTasks("REDUCE", "FAILED");
        this.printTasks("REDUCE", "KILLED");
        if (this.printAll) {
            this.printTasks("MAP", "SUCCESS");
            this.printTasks("REDUCE", "SUCCESS");
            this.printAllTaskAttempts("MAP");
            this.printAllTaskAttempts("REDUCE");
        }
        DefaultJobHistoryParser.NodesFilter filter = new DefaultJobHistoryParser.FailedOnNodesFilter();
        this.printFailedAttempts(filter);
        filter = new DefaultJobHistoryParser.KilledOnNodesFilter();
        this.printFailedAttempts(filter);
    }

    private void printJobDetails() {
        StringBuffer jobDetails = new StringBuffer();
        jobDetails.append("\nHadoop job: ").append(this.jobId);
        jobDetails.append("\n=====================================");
        jobDetails.append("\nJob tracker host name: ").append(this.trackerHostName);
        jobDetails.append("\njob tracker start time: ").append(new Date(Long.parseLong(this.trackerStartTime)));
        jobDetails.append("\nUser: ").append(this.job.get(JobHistory.Keys.USER));
        jobDetails.append("\nJobName: ").append(this.job.get(JobHistory.Keys.JOBNAME));
        jobDetails.append("\nJobConf: ").append(this.job.get(JobHistory.Keys.JOBCONF));
        jobDetails.append("\nSubmitted At: ").append(StringUtils.getFormattedTimeWithDiff(dateFormat, this.job.getLong(JobHistory.Keys.SUBMIT_TIME), 0L));
        jobDetails.append("\nLaunched At: ").append(StringUtils.getFormattedTimeWithDiff(dateFormat, this.job.getLong(JobHistory.Keys.LAUNCH_TIME), this.job.getLong(JobHistory.Keys.SUBMIT_TIME)));
        jobDetails.append("\nFinished At: ").append(StringUtils.getFormattedTimeWithDiff(dateFormat, this.job.getLong(JobHistory.Keys.FINISH_TIME), this.job.getLong(JobHistory.Keys.LAUNCH_TIME)));
        jobDetails.append("\nStatus: ").append(this.job.get(JobHistory.Keys.JOB_STATUS) == "" ? "Incomplete" : this.job.get(JobHistory.Keys.JOB_STATUS));
        jobDetails.append("\n=====================================");
        System.out.println(jobDetails.toString());
    }

    private void printTasks(String taskType, String taskStatus) {
        Map<String, JobHistory.Task> tasks = this.job.getAllTasks();
        StringBuffer taskList = new StringBuffer();
        taskList.append("\n").append(taskStatus).append(" ");
        taskList.append(taskType).append(" task list for ").append(this.jobId);
        taskList.append("\nTaskId\t\tStartTime\tFinishTime\tError");
        if (JobHistory.Values.MAP.name().equals(taskType)) {
            taskList.append("\tInputSplits");
        }
        taskList.append("\n====================================================");
        System.out.println(taskList.toString());
        for (JobHistory.Task task : tasks.values()) {
            if (!taskType.equals(task.get(JobHistory.Keys.TASK_TYPE)) || !taskStatus.equals(task.get(JobHistory.Keys.TASK_STATUS)) && !taskStatus.equals("all")) continue;
            taskList.setLength(0);
            taskList.append(task.get(JobHistory.Keys.TASKID));
            taskList.append("\t").append(StringUtils.getFormattedTimeWithDiff(dateFormat, task.getLong(JobHistory.Keys.START_TIME), 0L));
            taskList.append("\t").append(StringUtils.getFormattedTimeWithDiff(dateFormat, task.getLong(JobHistory.Keys.FINISH_TIME), task.getLong(JobHistory.Keys.START_TIME)));
            taskList.append("\t").append(task.get(JobHistory.Keys.ERROR));
            if (JobHistory.Values.MAP.name().equals(taskType)) {
                taskList.append("\t").append(task.get(JobHistory.Keys.SPLITS));
            }
            System.out.println(taskList.toString());
        }
    }

    private void printAllTaskAttempts(String taskType) {
        Map<String, JobHistory.Task> tasks = this.job.getAllTasks();
        StringBuffer taskList = new StringBuffer();
        taskList.append("\n").append(taskType);
        taskList.append(" task list for ").append(this.jobId);
        taskList.append("\nTaskId\t\tStartTime");
        if (JobHistory.Values.REDUCE.name().equals(taskType)) {
            taskList.append("\tShuffleFinished\tSortFinished");
        }
        taskList.append("\tFinishTime\tHostName\tError");
        taskList.append("\n====================================================");
        System.out.println(taskList.toString());
        for (JobHistory.Task task : tasks.values()) {
            for (JobHistory.TaskAttempt attempt : task.getTaskAttempts().values()) {
                if (!taskType.equals(task.get(JobHistory.Keys.TASK_TYPE))) continue;
                taskList.setLength(0);
                taskList.append(attempt.get(JobHistory.Keys.TASK_ATTEMPT_ID)).append("\t");
                taskList.append(StringUtils.getFormattedTimeWithDiff(dateFormat, attempt.getLong(JobHistory.Keys.START_TIME), 0L)).append("\t");
                if (JobHistory.Values.REDUCE.name().equals(taskType)) {
                    JobHistory.ReduceAttempt reduceAttempt = (JobHistory.ReduceAttempt)attempt;
                    taskList.append(StringUtils.getFormattedTimeWithDiff(dateFormat, reduceAttempt.getLong(JobHistory.Keys.SHUFFLE_FINISHED), reduceAttempt.getLong(JobHistory.Keys.START_TIME)));
                    taskList.append("\t");
                    taskList.append(StringUtils.getFormattedTimeWithDiff(dateFormat, reduceAttempt.getLong(JobHistory.Keys.SORT_FINISHED), reduceAttempt.getLong(JobHistory.Keys.SHUFFLE_FINISHED)));
                }
                taskList.append(StringUtils.getFormattedTimeWithDiff(dateFormat, attempt.getLong(JobHistory.Keys.FINISH_TIME), attempt.getLong(JobHistory.Keys.START_TIME)));
                taskList.append("\t");
                taskList.append(attempt.get(JobHistory.Keys.HOSTNAME)).append("\t");
                taskList.append(attempt.get(JobHistory.Keys.ERROR));
                System.out.println(taskList.toString());
            }
        }
    }

    private void printTaskSummary() {
        Map<String, JobHistory.Task> tasks = this.job.getAllTasks();
        int totalMaps = 0;
        int totalReduces = 0;
        int numFailedMaps = 0;
        int numKilledMaps = 0;
        int numFailedReduces = 0;
        int numKilledReduces = 0;
        long mapStarted = 0L;
        long mapFinished = 0L;
        long reduceStarted = 0L;
        long reduceFinished = 0L;
        TreeMap<String, String> allHosts = new TreeMap<String, String>();
        for (JobHistory.Task task : tasks.values()) {
            Map<String, JobHistory.TaskAttempt> attempts = task.getTaskAttempts();
            allHosts.put(task.get(JobHistory.Keys.HOSTNAME), "");
            for (JobHistory.TaskAttempt attempt : attempts.values()) {
                long startTime = attempt.getLong(JobHistory.Keys.START_TIME);
                long finishTime = attempt.getLong(JobHistory.Keys.FINISH_TIME);
                if (JobHistory.Values.MAP.name().equals(task.get(JobHistory.Keys.TASK_TYPE))) {
                    if (mapStarted == 0L || mapStarted > startTime) {
                        mapStarted = startTime;
                    }
                    if (mapFinished < finishTime) {
                        mapFinished = finishTime;
                    }
                    ++totalMaps;
                    if (JobHistory.Values.FAILED.name().equals(attempt.get(JobHistory.Keys.TASK_STATUS))) {
                        ++numFailedMaps;
                        continue;
                    }
                    if (!JobHistory.Values.KILLED.name().equals(attempt.get(JobHistory.Keys.TASK_STATUS))) continue;
                    ++numKilledMaps;
                    continue;
                }
                if (reduceStarted == 0L || reduceStarted > startTime) {
                    reduceStarted = startTime;
                }
                if (reduceFinished < finishTime) {
                    reduceFinished = finishTime;
                }
                ++totalReduces;
                if (JobHistory.Values.FAILED.name().equals(attempt.get(JobHistory.Keys.TASK_STATUS))) {
                    ++numFailedReduces;
                    continue;
                }
                if (!JobHistory.Values.KILLED.name().equals(attempt.get(JobHistory.Keys.TASK_STATUS))) continue;
                ++numKilledReduces;
            }
        }
        StringBuffer taskSummary = new StringBuffer();
        taskSummary.append("\nTask Summary");
        taskSummary.append("\n============================");
        taskSummary.append("\nKind\tTotal\t");
        taskSummary.append("Successful\tFailed\tKilled\tStartTime\tFinishTime");
        taskSummary.append("\n");
        taskSummary.append("\nMap\t").append(totalMaps);
        taskSummary.append("\t").append(this.job.getInt(JobHistory.Keys.FINISHED_MAPS));
        taskSummary.append("\t\t").append(numFailedMaps);
        taskSummary.append("\t").append(numKilledMaps);
        taskSummary.append("\t").append(StringUtils.getFormattedTimeWithDiff(dateFormat, mapStarted, 0L));
        taskSummary.append("\t").append(StringUtils.getFormattedTimeWithDiff(dateFormat, mapFinished, mapStarted));
        taskSummary.append("\nReduce\t").append(totalReduces);
        taskSummary.append("\t").append(this.job.getInt(JobHistory.Keys.FINISHED_REDUCES));
        taskSummary.append("\t\t").append(numFailedReduces);
        taskSummary.append("\t").append(numKilledReduces);
        taskSummary.append("\t").append(StringUtils.getFormattedTimeWithDiff(dateFormat, reduceStarted, 0L));
        taskSummary.append("\t").append(StringUtils.getFormattedTimeWithDiff(dateFormat, reduceFinished, reduceStarted));
        taskSummary.append("\n============================\n");
        System.out.println(taskSummary.toString());
    }

    private void printFailedAttempts(DefaultJobHistoryParser.NodesFilter filter) throws IOException {
        JobHistory.parseHistoryFromFS(this.jobLogFile, filter, this.fs);
        Map<String, Set<String>> badNodes = filter.getValues();
        StringBuffer attempts = new StringBuffer();
        if (badNodes.size() > 0) {
            attempts.append("\n").append(filter.getFailureType());
            attempts.append(" task attempts by nodes");
            attempts.append("\nHostname\tFailedTasks");
            attempts.append("\n===============================");
            System.out.println(attempts.toString());
            for (Map.Entry<String, Set<String>> entry : badNodes.entrySet()) {
                String node = entry.getKey();
                Set<String> failedTasks = entry.getValue();
                attempts.setLength(0);
                attempts.append(node).append("\t");
                for (String t : failedTasks) {
                    attempts.append(t).append(", ");
                }
                System.out.println(attempts.toString());
            }
        }
    }

    private void printJobAnalysis() {
        if (!JobHistory.Values.SUCCESS.name().equals(this.job.get(JobHistory.Keys.JOB_STATUS))) {
            System.out.println("No Analysis available as job did not finish");
            return;
        }
        Map<String, JobHistory.Task> tasks = this.job.getAllTasks();
        int finishedMaps = this.job.getInt(JobHistory.Keys.FINISHED_MAPS);
        int finishedReduces = this.job.getInt(JobHistory.Keys.FINISHED_REDUCES);
        JobHistory.Task[] mapTasks = new JobHistory.Task[finishedMaps];
        JobHistory.Task[] reduceTasks = new JobHistory.Task[finishedReduces];
        int mapIndex = 0;
        int reduceIndex = 0;
        long avgMapTime = 0L;
        long avgReduceTime = 0L;
        long avgShuffleTime = 0L;
        block0: for (JobHistory.Task task : tasks.values()) {
            Map<String, JobHistory.TaskAttempt> attempts = task.getTaskAttempts();
            for (JobHistory.TaskAttempt attempt : attempts.values()) {
                if (!attempt.get(JobHistory.Keys.TASK_STATUS).equals(JobHistory.Values.SUCCESS.name())) continue;
                long avgFinishTime = attempt.getLong(JobHistory.Keys.FINISH_TIME) - attempt.getLong(JobHistory.Keys.START_TIME);
                if (JobHistory.Values.MAP.name().equals(task.get(JobHistory.Keys.TASK_TYPE))) {
                    mapTasks[mapIndex++] = attempt;
                    avgMapTime += avgFinishTime;
                    continue block0;
                }
                reduceTasks[reduceIndex++] = attempt;
                avgShuffleTime += attempt.getLong(JobHistory.Keys.SHUFFLE_FINISHED) - attempt.getLong(JobHistory.Keys.START_TIME);
                avgReduceTime += attempt.getLong(JobHistory.Keys.FINISH_TIME) - attempt.getLong(JobHistory.Keys.SHUFFLE_FINISHED);
                continue block0;
            }
        }
        if (finishedMaps > 0) {
            avgMapTime /= (long)finishedMaps;
        }
        if (finishedReduces > 0) {
            avgReduceTime /= (long)finishedReduces;
            avgShuffleTime /= (long)finishedReduces;
        }
        System.out.println("\nAnalysis");
        System.out.println("=========");
        this.printAnalysis(mapTasks, this.cMap, "map", avgMapTime, 10);
        this.printLast(mapTasks, "map", this.cFinishMapRed);
        if (reduceTasks.length > 0) {
            this.printAnalysis(reduceTasks, this.cShuffle, "shuffle", avgShuffleTime, 10);
            this.printLast(reduceTasks, "shuffle", this.cFinishShuffle);
            this.printAnalysis(reduceTasks, this.cReduce, "reduce", avgReduceTime, 10);
            this.printLast(reduceTasks, "reduce", this.cFinishMapRed);
        }
        System.out.println("=========");
    }

    private void printLast(JobHistory.Task[] tasks, String taskType, Comparator<JobHistory.Task> cmp) {
        Arrays.sort(tasks, this.cFinishMapRed);
        JobHistory.Task last = tasks[0];
        StringBuffer lastBuf = new StringBuffer();
        lastBuf.append("The last ").append(taskType);
        lastBuf.append(" task ").append(last.get(JobHistory.Keys.TASKID));
        Long finishTime = "shuffle".equals(taskType) ? Long.valueOf(last.getLong(JobHistory.Keys.SHUFFLE_FINISHED)) : Long.valueOf(last.getLong(JobHistory.Keys.FINISH_TIME));
        lastBuf.append(" finished at (relative to the Job launch time): ");
        lastBuf.append(StringUtils.getFormattedTimeWithDiff(dateFormat, finishTime, this.job.getLong(JobHistory.Keys.LAUNCH_TIME)));
        System.out.println(lastBuf.toString());
    }

    private void printAnalysis(JobHistory.Task[] tasks, Comparator<JobHistory.Task> cmp, String taskType, long avg, int showTasks) {
        Arrays.sort(tasks, cmp);
        JobHistory.Task min = tasks[tasks.length - 1];
        StringBuffer details = new StringBuffer();
        details.append("\nTime taken by best performing ");
        details.append(taskType).append(" task ");
        details.append(min.get(JobHistory.Keys.TASKID)).append(": ");
        if ("map".equals(taskType)) {
            details.append(StringUtils.formatTimeDiff(min.getLong(JobHistory.Keys.FINISH_TIME), min.getLong(JobHistory.Keys.START_TIME)));
        } else if ("shuffle".equals(taskType)) {
            details.append(StringUtils.formatTimeDiff(min.getLong(JobHistory.Keys.SHUFFLE_FINISHED), min.getLong(JobHistory.Keys.START_TIME)));
        } else {
            details.append(StringUtils.formatTimeDiff(min.getLong(JobHistory.Keys.FINISH_TIME), min.getLong(JobHistory.Keys.SHUFFLE_FINISHED)));
        }
        details.append("\nAverage time taken by ");
        details.append(taskType).append(" tasks: ");
        details.append(StringUtils.formatTimeDiff(avg, 0L));
        details.append("\nWorse performing ");
        details.append(taskType).append(" tasks: ");
        details.append("\nTaskId\t\tTimetaken");
        System.out.println(details.toString());
        for (int i = 0; i < showTasks && i < tasks.length; ++i) {
            details.setLength(0);
            details.append(tasks[i].get(JobHistory.Keys.TASKID)).append(" ");
            if ("map".equals(taskType)) {
                details.append(StringUtils.formatTimeDiff(tasks[i].getLong(JobHistory.Keys.FINISH_TIME), tasks[i].getLong(JobHistory.Keys.START_TIME)));
            } else if ("shuffle".equals(taskType)) {
                details.append(StringUtils.formatTimeDiff(tasks[i].getLong(JobHistory.Keys.SHUFFLE_FINISHED), tasks[i].getLong(JobHistory.Keys.START_TIME)));
            } else {
                details.append(StringUtils.formatTimeDiff(tasks[i].getLong(JobHistory.Keys.FINISH_TIME), tasks[i].getLong(JobHistory.Keys.SHUFFLE_FINISHED)));
            }
            System.out.println(details.toString());
        }
    }
}

