/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.net.InetSocketAddress;
import java.net.URL;
import java.nio.channels.FileChannel;
import java.util.Random;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import junit.framework.TestCase;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.FSConstants;
import org.apache.hadoop.hdfs.protocol.LocatedBlocks;
import org.apache.hadoop.io.IOUtils;

public class TestDatanodeBlockScanner
extends TestCase {
    private static final Log LOG = LogFactory.getLog(TestDatanodeBlockScanner.class);
    private static Pattern pattern = Pattern.compile(".*?(blk_[-]*\\d+).*?scan time\\s*:\\s*(\\d+)");

    private static long waitForVerification(DatanodeInfo dn, FileSystem fs, Path file) throws IOException {
        URL url = new URL("http://localhost:" + dn.getInfoPort() + "/blockScannerReport?listblocks");
        long lastWarnTime = System.currentTimeMillis();
        long verificationTime = 0L;
        String block = DFSTestUtil.getFirstBlock(fs, file).getBlockName();
        while (verificationTime <= 0L) {
            String response = DFSTestUtil.urlGet(url);
            Matcher matcher = pattern.matcher(response);
            while (matcher.find()) {
                if (!block.equals(matcher.group(1))) continue;
                verificationTime = Long.parseLong(matcher.group(2));
                break;
            }
            if (verificationTime > 0L) continue;
            long now = System.currentTimeMillis();
            if (now - lastWarnTime >= 5000L) {
                LOG.info((Object)("Waiting for verification of " + block));
                lastWarnTime = now;
            }
            try {
                Thread.sleep(500L);
            }
            catch (InterruptedException ignored) {}
        }
        return verificationTime;
    }

    public void testDatanodeBlockScanner() throws IOException {
        long startTime = System.currentTimeMillis();
        Configuration conf = new Configuration();
        MiniDFSCluster cluster = new MiniDFSCluster(conf, 1, true, null);
        cluster.waitActive();
        FileSystem fs = cluster.getFileSystem();
        Path file1 = new Path("/tmp/testBlockVerification/file1");
        Path file2 = new Path("/tmp/testBlockVerification/file2");
        DFSTestUtil.createFile(fs, file1, 10L, (short)1, 0L);
        cluster.shutdown();
        cluster = new MiniDFSCluster(conf, 1, false, null);
        cluster.waitActive();
        DFSClient dfsClient = new DFSClient(new InetSocketAddress("localhost", cluster.getNameNodePort()), conf);
        fs = cluster.getFileSystem();
        DatanodeInfo dn = dfsClient.datanodeReport(FSConstants.DatanodeReportType.LIVE)[0];
        TestDatanodeBlockScanner.assertTrue((TestDatanodeBlockScanner.waitForVerification(dn, fs, file1) > startTime ? 1 : 0) != 0);
        DFSTestUtil.createFile(fs, file2, 10L, (short)1, 0L);
        IOUtils.copyBytes((InputStream)fs.open(file2), (OutputStream)new IOUtils.NullOutputStream(), (Configuration)conf, (boolean)true);
        TestDatanodeBlockScanner.assertTrue((TestDatanodeBlockScanner.waitForVerification(dn, fs, file2) > startTime ? 1 : 0) != 0);
        cluster.shutdown();
    }

    public static boolean corruptReplica(String blockName, int replica) throws IOException {
        Random random = new Random();
        File baseDir = new File(System.getProperty("test.build.data"), "dfs/data");
        boolean corrupted = false;
        for (int i = replica * 2; i < replica * 2 + 2; ++i) {
            File blockFile = new File(baseDir, "data" + (i + 1) + "/current/" + blockName);
            if (!blockFile.exists()) continue;
            RandomAccessFile raFile = new RandomAccessFile(blockFile, "rw");
            FileChannel channel = raFile.getChannel();
            String badString = "BADBAD";
            int rand = random.nextInt((int)channel.size() / 2);
            raFile.seek(rand);
            raFile.write(badString.getBytes());
            raFile.close();
            corrupted = true;
        }
        return corrupted;
    }

    public void testBlockCorruptionPolicy() throws IOException {
        Configuration conf = new Configuration();
        conf.setLong("dfs.blockreport.intervalMsec", 1000L);
        Random random = new Random();
        FileSystem fs = null;
        DFSClient dfsClient = null;
        LocatedBlocks blocks = null;
        int blockCount = 0;
        int rand = random.nextInt(3);
        MiniDFSCluster cluster = new MiniDFSCluster(conf, 3, true, null);
        cluster.waitActive();
        fs = cluster.getFileSystem();
        Path file1 = new Path("/tmp/testBlockVerification/file1");
        DFSTestUtil.createFile(fs, file1, 1024L, (short)3, 0L);
        String block = DFSTestUtil.getFirstBlock(fs, file1).getBlockName();
        dfsClient = new DFSClient(new InetSocketAddress("localhost", cluster.getNameNodePort()), conf);
        do {
            blocks = dfsClient.namenode.getBlockLocations(file1.toString(), 0L, Long.MAX_VALUE);
            blockCount = blocks.get(0).getLocations().length;
            try {
                LOG.info((Object)"Looping until expected blockCount of 3 is received");
                Thread.sleep(1000L);
            }
            catch (InterruptedException ignore) {
                // empty catch block
            }
        } while (blockCount != 3);
        TestDatanodeBlockScanner.assertTrue((!blocks.get(0).isCorrupt() ? 1 : 0) != 0);
        TestDatanodeBlockScanner.corruptReplica(block, rand);
        cluster.restartDataNode(rand);
        do {
            blocks = dfsClient.namenode.getBlockLocations(file1.toString(), 0L, Long.MAX_VALUE);
            blockCount = blocks.get(0).getLocations().length;
            try {
                LOG.info((Object)"Looping until expected blockCount of 2 is received");
                Thread.sleep(1000L);
            }
            catch (InterruptedException ignore) {
                // empty catch block
            }
        } while (blockCount != 2);
        TestDatanodeBlockScanner.assertTrue((!blocks.get(0).isCorrupt() ? 1 : 0) != 0);
        TestDatanodeBlockScanner.corruptReplica(block, 0);
        TestDatanodeBlockScanner.corruptReplica(block, 1);
        TestDatanodeBlockScanner.corruptReplica(block, 2);
        try {
            IOUtils.copyBytes((InputStream)fs.open(file1), (OutputStream)new IOUtils.NullOutputStream(), (Configuration)conf, (boolean)true);
        }
        catch (IOException e) {
            // empty catch block
        }
        do {
            blocks = dfsClient.namenode.getBlockLocations(file1.toString(), 0L, Long.MAX_VALUE);
            blockCount = blocks.get(0).getLocations().length;
            try {
                LOG.info((Object)"Looping until expected blockCount of 3 is received");
                Thread.sleep(1000L);
            }
            catch (InterruptedException ignore) {
                // empty catch block
            }
        } while (blockCount != 3);
        TestDatanodeBlockScanner.assertTrue((blocks.get(0).isCorrupt() ? 1 : 0) != 0);
        cluster.shutdown();
    }

    public void testBlockCorruptionRecoveryPolicy() throws IOException {
        LOG.info((Object)"Testing corrupt replica recovery for one corrupt replica");
        this.blockCorruptionRecoveryPolicy(4, (short)3, 1);
        LOG.info((Object)"Testing corrupt replica recovery for two corrupt replicas");
        this.blockCorruptionRecoveryPolicy(5, (short)3, 2);
    }

    private void blockCorruptionRecoveryPolicy(int numDataNodes, short numReplicas, int numCorruptReplicas) throws IOException {
        int i;
        Configuration conf = new Configuration();
        conf.setLong("dfs.blockreport.intervalMsec", 30L);
        conf.setLong("dfs.replication.interval", 30L);
        conf.setLong("dfs.heartbeat.interval", 30L);
        conf.setBoolean("dfs.replication.considerLoad", false);
        Random random = new Random();
        FileSystem fs = null;
        DFSClient dfsClient = null;
        LocatedBlocks blocks = null;
        int replicaCount = 0;
        int rand = random.nextInt(numDataNodes);
        MiniDFSCluster cluster = new MiniDFSCluster(conf, numDataNodes, true, null);
        cluster.waitActive();
        fs = cluster.getFileSystem();
        Path file1 = new Path("/tmp/testBlockCorruptRecovery/file");
        DFSTestUtil.createFile(fs, file1, 1024L, numReplicas, 0L);
        Block blk = DFSTestUtil.getFirstBlock(fs, file1);
        String block = blk.getBlockName();
        dfsClient = new DFSClient(new InetSocketAddress("localhost", cluster.getNameNodePort()), conf);
        blocks = dfsClient.namenode.getBlockLocations(file1.toString(), 0L, Long.MAX_VALUE);
        replicaCount = blocks.get(0).getLocations().length;
        while (replicaCount != numReplicas) {
            try {
                LOG.info((Object)("Looping until expected replicaCount of " + numReplicas + "is reached"));
                Thread.sleep(1000L);
            }
            catch (InterruptedException ignore) {
                // empty catch block
            }
            blocks = dfsClient.namenode.getBlockLocations(file1.toString(), 0L, Long.MAX_VALUE);
            replicaCount = blocks.get(0).getLocations().length;
        }
        TestDatanodeBlockScanner.assertTrue((!blocks.get(0).isCorrupt() ? 1 : 0) != 0);
        int[] corruptReplicasDNIDs = new int[numCorruptReplicas];
        int j = 0;
        for (i = 0; j != numCorruptReplicas && i < numDataNodes; ++i) {
            if (!TestDatanodeBlockScanner.corruptReplica(block, i)) continue;
            corruptReplicasDNIDs[j++] = i;
        }
        for (i = 0; i < numCorruptReplicas; ++i) {
            cluster.restartDataNode(corruptReplicasDNIDs[i]);
        }
        int corruptReplicaSize = cluster.getNameNode().namesystem.corruptReplicas.numCorruptReplicas(blk);
        while (corruptReplicaSize != numCorruptReplicas) {
            try {
                IOUtils.copyBytes((InputStream)fs.open(file1), (OutputStream)new IOUtils.NullOutputStream(), (Configuration)conf, (boolean)true);
            }
            catch (IOException e) {
                // empty catch block
            }
            try {
                LOG.info((Object)("Looping until expected " + numCorruptReplicas + " are " + "reported. Current reported " + corruptReplicaSize));
                Thread.sleep(1000L);
            }
            catch (InterruptedException ignore) {
                // empty catch block
            }
            corruptReplicaSize = cluster.getNameNode().namesystem.corruptReplicas.numCorruptReplicas(blk);
        }
        blocks = dfsClient.namenode.getBlockLocations(file1.toString(), 0L, Long.MAX_VALUE);
        replicaCount = blocks.get(0).getLocations().length;
        while (replicaCount != numReplicas) {
            try {
                LOG.info((Object)("Looping until block gets rereplicated to " + numReplicas));
                Thread.sleep(1000L);
            }
            catch (InterruptedException ignore) {
                // empty catch block
            }
            blocks = dfsClient.namenode.getBlockLocations(file1.toString(), 0L, Long.MAX_VALUE);
            replicaCount = blocks.get(0).getLocations().length;
        }
        corruptReplicaSize = cluster.getNameNode().namesystem.corruptReplicas.numCorruptReplicas(blk);
        while (corruptReplicaSize != 0 || replicaCount != numReplicas) {
            try {
                LOG.info((Object)"Looping until corrupt replica is invalidated");
                Thread.sleep(1000L);
            }
            catch (InterruptedException ignore) {
                // empty catch block
            }
            corruptReplicaSize = cluster.getNameNode().namesystem.corruptReplicas.numCorruptReplicas(blk);
            blocks = dfsClient.namenode.getBlockLocations(file1.toString(), 0L, Long.MAX_VALUE);
            replicaCount = blocks.get(0).getLocations().length;
        }
        TestDatanodeBlockScanner.assertTrue((corruptReplicaSize == 0 ? 1 : 0) != 0);
        TestDatanodeBlockScanner.assertTrue((replicaCount == numReplicas ? 1 : 0) != 0);
        TestDatanodeBlockScanner.assertTrue((!blocks.get(0).isCorrupt() ? 1 : 0) != 0);
        cluster.shutdown();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void testTruncatedBlockReport() throws Exception {
        Configuration conf = new Configuration();
        int REPLICATION_FACTOR = 2;
        MiniDFSCluster cluster = new MiniDFSCluster(conf, 2, true, null);
        cluster.waitActive();
        FileSystem fs = cluster.getFileSystem();
        try {
            Path fileName = new Path("/file1");
            DFSTestUtil.createFile(fs, fileName, 1L, (short)2, 0L);
            DFSTestUtil.waitReplication(fs, fileName, (short)2);
            String block = DFSTestUtil.getFirstBlock(fs, fileName).getBlockName();
            TestDatanodeBlockScanner.changeReplicaLength(block, 0, -1);
            cluster.shutdown();
            cluster = new MiniDFSCluster(0, conf, 2, false, true, null, null, null);
            cluster.startDataNodes(conf, 1, true, null, null);
            cluster.waitActive();
            DFSTestUtil.waitReplication(cluster.getFileSystem(), fileName, (short)2);
            TestDatanodeBlockScanner.waitForBlockDeleted(block, 0);
        }
        finally {
            cluster.shutdown();
        }
    }

    static boolean changeReplicaLength(String blockName, int dnIndex, int lenDelta) throws IOException {
        File baseDir = new File(System.getProperty("test.build.data"), "dfs/data");
        for (int i = dnIndex * 2; i < dnIndex * 2 + 2; ++i) {
            File blockFile = new File(baseDir, "data" + (i + 1) + "/current/" + blockName);
            if (!blockFile.exists()) continue;
            RandomAccessFile raFile = new RandomAccessFile(blockFile, "rw");
            raFile.setLength(raFile.length() + (long)lenDelta);
            raFile.close();
            return true;
        }
        return false;
    }

    private static void waitForBlockDeleted(String blockName, int dnIndex) throws IOException, InterruptedException {
        File baseDir = new File(System.getProperty("test.build.data"), "dfs/data");
        File blockFile1 = new File(baseDir, "data" + (2 * dnIndex + 1) + "/current/" + blockName);
        File blockFile2 = new File(baseDir, "data" + (2 * dnIndex + 2) + "/current/" + blockName);
        while (blockFile1.exists() || blockFile2.exists()) {
            Thread.sleep(100L);
        }
    }
}

