/*
 * Decompiled with CFR 0.152.
 */
package edu.berkeley.xtrace.server;

import edu.berkeley.xtrace.TaskID;
import edu.berkeley.xtrace.XTraceException;
import edu.berkeley.xtrace.reporting.Report;
import edu.berkeley.xtrace.server.QueryableReportStore;
import edu.berkeley.xtrace.server.ReportSource;
import edu.berkeley.xtrace.server.TaskRecord;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.net.URLDecoder;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.log4j.Logger;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import org.mortbay.jetty.Server;
import org.mortbay.jetty.servlet.Context;
import org.mortbay.jetty.servlet.DefaultServlet;
import org.mortbay.jetty.servlet.ServletHolder;
import org.mortbay.servlet.CGI;

public final class XTraceServer {
    private static final Logger LOG = Logger.getLogger(XTraceServer.class);
    private static ReportSource[] sources;
    private static BlockingQueue<String> incomingReportQueue;
    private static BlockingQueue<String> reportsToStorageQueue;
    private static ThreadPerTaskExecutor sourcesExecutor;
    private static ExecutorService storeExecutor;
    private static QueryableReportStore reportstore;
    private static final DateFormat JSON_DATE_FORMAT;
    private static final DateFormat HTML_DATE_FORMAT;
    private static final int PAGE_LENGTH = 25;

    public static void main(String[] args) {
        if (System.getProperty("xtrace.server.store") == null) {
            if (args.length < 1) {
                System.err.println("Usage: XTraceServer <dataDir>");
                System.exit(1);
            }
            System.setProperty("xtrace.server.storedirectory", args[0]);
        }
        XTraceServer.setupReportSources();
        XTraceServer.setupReportStore();
        XTraceServer.setupBackplane();
        XTraceServer.setupWebInterface();
    }

    private static void setupReportSources() {
        incomingReportQueue = new ArrayBlockingQueue<String>(1024, true);
        sourcesExecutor = new ThreadPerTaskExecutor();
        String sourcesStr = "edu.berkeley.xtrace.server.UdpReportSource,edu.berkeley.xtrace.server.TcpReportSource,edu.berkeley.xtrace.server.ThriftReportSource";
        if (System.getProperty("xtrace.server.sources") != null) {
            sourcesStr = System.getProperty("xtrace.server.sources");
        } else {
            LOG.warn("No server report sources specified... using defaults (Udp,Tcp,Thrift)");
        }
        String[] sourcesLst = sourcesStr.split(",");
        sources = new ReportSource[sourcesLst.length];
        for (int i = 0; i < sourcesLst.length; ++i) {
            try {
                LOG.info("Starting report source '" + sourcesLst[i] + "'");
                XTraceServer.sources[i] = (ReportSource)Class.forName(sourcesLst[i]).newInstance();
            }
            catch (InstantiationException e1) {
                LOG.fatal("Could not instantiate report source", e1);
                System.exit(-1);
            }
            catch (IllegalAccessException e1) {
                LOG.fatal("Could not access report source", e1);
                System.exit(-1);
            }
            catch (ClassNotFoundException e1) {
                LOG.fatal("Could not find report source class", e1);
                System.exit(-1);
            }
            sources[i].setReportQueue(incomingReportQueue);
            try {
                sources[i].initialize();
            }
            catch (XTraceException e) {
                LOG.warn("Unable to initialize report source", e);
                System.exit(-1);
            }
            sourcesExecutor.execute(sources[i]);
        }
    }

    private static void setupReportStore() {
        reportsToStorageQueue = new ArrayBlockingQueue<String>(1024);
        String storeStr = "edu.berkeley.xtrace.server.FileTreeReportStore";
        if (System.getProperty("xtrace.server.store") != null) {
            storeStr = System.getProperty("xtrace.server.store");
        } else {
            LOG.warn("No server report store specified... using default (FileTreeReportStore)");
        }
        reportstore = null;
        try {
            reportstore = (QueryableReportStore)Class.forName(storeStr).newInstance();
        }
        catch (InstantiationException e1) {
            LOG.fatal("Could not instantiate report store", e1);
            System.exit(-1);
        }
        catch (IllegalAccessException e1) {
            LOG.fatal("Could not access report store class", e1);
            System.exit(-1);
        }
        catch (ClassNotFoundException e1) {
            LOG.fatal("Could not find report store class", e1);
            System.exit(-1);
        }
        reportstore.setReportQueue(reportsToStorageQueue);
        try {
            reportstore.initialize();
        }
        catch (XTraceException e) {
            LOG.fatal("Unable to start report store", e);
            System.exit(-1);
        }
        storeExecutor = Executors.newSingleThreadExecutor();
        storeExecutor.execute(reportstore);
        String syncIntervalStr = System.getProperty("xtrace.server.syncinterval", "5");
        long syncInterval = Integer.parseInt(syncIntervalStr);
        Timer timer = new Timer();
        timer.schedule((TimerTask)new SyncTimer(reportstore), syncInterval * 1000L, syncInterval * 1000L);
        Runtime.getRuntime().addShutdownHook(new Thread(){

            @Override
            public void run() {
                reportstore.shutdown();
            }
        });
    }

    private static void setupBackplane() {
        new Thread(new Runnable(){

            @Override
            public void run() {
                LOG.info("Backplane waiting for packets");
                while (true) {
                    String msg = null;
                    try {
                        msg = (String)incomingReportQueue.take();
                    }
                    catch (InterruptedException e) {
                        LOG.warn("Interrupted", e);
                        continue;
                    }
                    reportsToStorageQueue.offer(msg);
                }
            }
        }).start();
    }

    private static void setupWebInterface() {
        String webDir = System.getProperty("xtrace.backend.webui.dir");
        if (webDir == null) {
            LOG.warn("No webui directory specified... using default (./src/webui)");
            webDir = "./src/webui";
        }
        int httpPort = Integer.parseInt(System.getProperty("xtrace.backend.httpport", "8080"));
        try {
            Velocity.setProperty("runtime.log.logsystem.class", "org.apache.velocity.runtime.log.Log4JLogChute");
            Velocity.setProperty("runtime.log.logsystem.log4j.logger", "edu.berkeley.xtrace.server.XTraceServer");
            Velocity.setProperty("file.resource.loader.path", webDir + "/templates");
            Velocity.setProperty("file.resource.loader.cache", "true");
            Velocity.init();
        }
        catch (Exception e) {
            LOG.warn("Failed to initialize Velocity", e);
        }
        Server server = new Server(httpPort);
        Context context = new Context(server, "/");
        ServletHolder cgiHolder = new ServletHolder(new CGI());
        cgiHolder.setInitParameter("cgibinResourceBase", webDir + "/cgi-bin");
        if (System.getenv("PATH") != null) {
            cgiHolder.setInitParameter("Path", System.getenv("PATH"));
        }
        context.addServlet(cgiHolder, "*.cgi");
        context.addServlet(cgiHolder, "*.pl");
        context.addServlet(cgiHolder, "*.py");
        context.addServlet(cgiHolder, "*.rb");
        context.addServlet(cgiHolder, "*.tcl");
        context.addServlet(new ServletHolder(new GetReportsServlet()), "/reports/*");
        context.addServlet(new ServletHolder(new GetLatestTaskServlet()), "/latestTask");
        context.addServlet(new ServletHolder(new TagServlet()), "/tag/*");
        context.addServlet(new ServletHolder(new TitleServlet()), "/title/*");
        context.addServlet(new ServletHolder(new TitleLikeServlet()), "/titleLike/*");
        context.setResourceBase(webDir + "/html");
        context.addServlet(new ServletHolder(new IndexServlet()), "/");
        try {
            server.start();
        }
        catch (Exception e) {
            LOG.warn("Unable to start web interface", e);
        }
    }

    private static String getUriPastServletName(HttpServletRequest request) {
        String text;
        String uri = request.getRequestURI();
        int pathLen = request.getServletPath().length() + 1;
        String string = text = uri.length() > pathLen ? uri.substring(pathLen) : null;
        if (text != null) {
            try {
                text = URLDecoder.decode(text, "UTF-8");
            }
            catch (UnsupportedEncodingException e) {
                e.printStackTrace();
                return null;
            }
        }
        return text;
    }

    private static void showTasks(HttpServletRequest request, HttpServletResponse response, Collection<TaskRecord> tasks, String title, boolean showDbStats) throws IOException {
        if ("json".equals(request.getParameter("format"))) {
            response.setContentType("text/plain");
        } else {
            response.setContentType("text/html");
        }
        int offset = XTraceServer.getOffset(request);
        int length = XTraceServer.getLength(request);
        VelocityContext context = new VelocityContext();
        context.put("tasks", tasks);
        context.put("title", title);
        context.put("reportStore", reportstore);
        context.put("request", request);
        context.put("offset", offset);
        context.put("length", length);
        context.put("lastResultNum", offset + length - 1);
        context.put("prevOffset", Math.max(0, offset - length));
        context.put("nextOffset", offset + length);
        context.put("showStats", showDbStats);
        context.put("JSON_DATE_FORMAT", JSON_DATE_FORMAT);
        context.put("HTML_DATE_FORMAT", HTML_DATE_FORMAT);
        context.put("PAGE_LENGTH", 25);
        try {
            Velocity.mergeTemplate("tasks.vm", "UTF-8", context, response.getWriter());
            response.setStatus(200);
        }
        catch (Exception e) {
            LOG.warn("Failed to display tasks.vm", e);
            response.sendError(500, "Failed to display tasks.vm");
        }
    }

    private static int getLength(HttpServletRequest request) {
        int length = XTraceServer.getIntParam(request, "length", 25);
        return Math.max(length, 0);
    }

    private static int getOffset(HttpServletRequest request) {
        int offset = XTraceServer.getIntParam(request, "offset", 0);
        return Math.max(offset, 0);
    }

    private static final int getIntParam(HttpServletRequest request, String name, int defaultValue) {
        try {
            return Integer.parseInt(request.getParameter(name));
        }
        catch (Exception ex) {
            return defaultValue;
        }
    }

    static {
        JSON_DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        HTML_DATE_FORMAT = new SimpleDateFormat("MMM dd yyyy, HH:mm:ss");
    }

    private static final class SyncTimer
    extends TimerTask {
        private QueryableReportStore reportstore;

        public SyncTimer(QueryableReportStore reportstore) {
            this.reportstore = reportstore;
        }

        @Override
        public void run() {
            this.reportstore.sync();
        }
    }

    private static class IndexServlet
    extends DefaultServlet {
        private IndexServlet() {
        }

        @Override
        protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
            if (request.getRequestURI().equals("/")) {
                List<TaskRecord> tasks = reportstore.getLatestTasks(XTraceServer.getOffset(request), XTraceServer.getLength(request));
                XTraceServer.showTasks(request, response, tasks, "X-Trace Latest Tasks", true);
            } else {
                super.doGet(request, response);
            }
        }
    }

    private static class TitleLikeServlet
    extends HttpServlet {
        private TitleLikeServlet() {
        }

        @Override
        protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
            String title = XTraceServer.getUriPastServletName(request);
            if (title == null || title.equalsIgnoreCase("")) {
                response.sendError(505, "No title given");
            } else {
                List<TaskRecord> taskInfos = reportstore.getTasksByTitleSubstring(title, XTraceServer.getOffset(request), XTraceServer.getLength(request));
                XTraceServer.showTasks(request, response, taskInfos, "Tasks with title like: " + title, false);
            }
        }
    }

    private static class TitleServlet
    extends HttpServlet {
        private TitleServlet() {
        }

        @Override
        protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
            String title = XTraceServer.getUriPastServletName(request);
            if (title == null || title.equalsIgnoreCase("")) {
                response.sendError(505, "No title given");
            } else {
                List<TaskRecord> taskInfos = reportstore.getTasksByTitle(title, XTraceServer.getOffset(request), XTraceServer.getLength(request));
                XTraceServer.showTasks(request, response, taskInfos, "Tasks with title: " + title, false);
            }
        }
    }

    private static class TagServlet
    extends HttpServlet {
        private TagServlet() {
        }

        @Override
        protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
            String tag = XTraceServer.getUriPastServletName(request);
            if (tag == null || tag.equalsIgnoreCase("")) {
                response.sendError(505, "No tag given");
            } else {
                List<TaskRecord> taskInfos = reportstore.getTasksByTag(tag, XTraceServer.getOffset(request), XTraceServer.getLength(request));
                XTraceServer.showTasks(request, response, taskInfos, "Tasks with tag: " + tag, false);
            }
        }
    }

    private static class GetLatestTaskServlet
    extends HttpServlet {
        private GetLatestTaskServlet() {
        }

        @Override
        protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
            response.setContentType("text/plain");
            response.setStatus(200);
            PrintWriter out = response.getWriter();
            List<TaskRecord> task = reportstore.getLatestTasks(0, 1);
            if (task.size() != 1) {
                LOG.warn("getLatestTasks(1) returned " + task.size() + " entries");
                return;
            }
            try {
                Iterator<Report> iter = reportstore.getReportsByTask(task.get(0).getTaskId());
                while (iter.hasNext()) {
                    Report r = iter.next();
                    ((Writer)out).write(r.toString());
                    ((Writer)out).write("\n");
                }
            }
            catch (XTraceException e) {
                LOG.warn("Internal error", e);
                ((Writer)out).write("Internal error: " + e);
            }
        }
    }

    private static class GetReportsServlet
    extends HttpServlet {
        private GetReportsServlet() {
        }

        @Override
        protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
            response.setContentType("text/plain");
            response.setStatus(200);
            String uri = request.getRequestURI();
            int pathLen = request.getServletPath().length() + 1;
            String taskId = uri.length() > pathLen ? uri.substring(pathLen) : null;
            PrintWriter out = response.getWriter();
            if (taskId != null) {
                Iterator<Report> iter;
                try {
                    iter = reportstore.getReportsByTask(TaskID.createFromString(taskId));
                }
                catch (XTraceException e) {
                    throw new ServletException(e);
                }
                while (iter.hasNext()) {
                    ((Writer)out).write(iter.next().toString());
                    ((Writer)out).write("\n");
                }
            }
        }
    }

    private static class ThreadPerTaskExecutor
    implements Executor {
        private ThreadPerTaskExecutor() {
        }

        @Override
        public void execute(Runnable r) {
            new Thread(r).start();
        }
    }
}

