/*
 * Decompiled with CFR 0.152.
 */
package edu.berkeley.xtrace.server;

import edu.berkeley.xtrace.TaskID;
import edu.berkeley.xtrace.XTraceException;
import edu.berkeley.xtrace.XTraceMetadata;
import edu.berkeley.xtrace.reporting.Report;
import edu.berkeley.xtrace.server.QueryableReportStore;
import edu.berkeley.xtrace.server.TaskRecord;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import java.util.concurrent.BlockingQueue;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.log4j.Logger;

public final class FileTreeReportStore
implements QueryableReportStore {
    private static final Logger LOG = Logger.getLogger(FileTreeReportStore.class);
    private String dataDirName;
    private File dataRootDir;
    private BlockingQueue<String> incomingReports;
    private LRUFileHandleCache fileCache;
    private Connection conn;
    private PreparedStatement countTasks;
    private PreparedStatement insert;
    private PreparedStatement update;
    private PreparedStatement updateTitle;
    private PreparedStatement updateTags;
    private PreparedStatement updatedSince;
    private PreparedStatement numByTask;
    private PreparedStatement getByTag;
    private PreparedStatement totalNumReports;
    private PreparedStatement totalNumTasks;
    private PreparedStatement lastUpdatedByTask;
    private PreparedStatement lastTasks;
    private PreparedStatement getTags;
    private PreparedStatement getByTitle;
    private PreparedStatement getByTitleApprox;
    private boolean shouldOperate = false;
    private boolean databaseInitialized = false;
    private static final Pattern XTRACE_LINE = Pattern.compile("^X-Trace:\\s+([0-9A-Fa-f]+)$", 8);

    @Override
    public synchronized void setReportQueue(BlockingQueue<String> q) {
        this.incomingReports = q;
    }

    @Override
    public synchronized void initialize() throws XTraceException {
        this.dataDirName = System.getProperty("xtrace.server.storedirectory");
        if (this.dataDirName == null) {
            throw new XTraceException("FileTreeReportStore selected, but no xtrace.server.storedirectory specified");
        }
        this.dataRootDir = new File(this.dataDirName);
        if (!this.dataRootDir.isDirectory()) {
            throw new XTraceException("Data Store location isn't a directory: " + this.dataDirName);
        }
        if (!this.dataRootDir.canWrite()) {
            throw new XTraceException("Can't write to data store directory");
        }
        this.fileCache = new LRUFileHandleCache(25, this.dataRootDir);
        this.initializeDatabase();
        this.shouldOperate = true;
    }

    private void initializeDatabase() throws XTraceException {
        System.setProperty("derby.system.home", this.dataDirName);
        try {
            Class.forName("org.apache.derby.jdbc.EmbeddedDriver").newInstance();
        }
        catch (InstantiationException e) {
            throw new XTraceException("Unable to instantiate internal database", e);
        }
        catch (IllegalAccessException e) {
            throw new XTraceException("Unable to access internal database class", e);
        }
        catch (ClassNotFoundException e) {
            throw new XTraceException("Unable to locate internal database class", e);
        }
        try {
            try {
                this.conn = DriverManager.getConnection("jdbc:derby:tasks");
            }
            catch (SQLException e) {
                this.conn = DriverManager.getConnection("jdbc:derby:tasks;create=true");
                this.createTables();
                this.conn.commit();
            }
            this.conn.setAutoCommit(false);
        }
        catch (SQLException e) {
            throw new XTraceException("Unable to connect to interal database: " + e.getSQLState(), e);
        }
        LOG.info("Successfully connected to the internal Derby database");
        try {
            this.createPreparedStatements();
        }
        catch (SQLException e) {
            throw new XTraceException("Unable to setup prepared statements", e);
        }
        this.databaseInitialized = true;
    }

    private void createTables() throws SQLException {
        Statement s = this.conn.createStatement();
        s.executeUpdate("create table tasks(taskId varchar(40) not null primary key, firstSeen timestamp default current_timestamp not null, lastUpdated timestamp default current_timestamp not null, numReports integer default 1 not null, tags varchar(512), title varchar(128))");
        s.executeUpdate("create index idx_tasks on tasks(taskid)");
        s.executeUpdate("create index idx_firstseen on tasks(firstSeen)");
        s.executeUpdate("create index idx_lastUpdated on tasks(lastUpdated)");
        s.executeUpdate("create index idx_tags on tasks(tags)");
        s.executeUpdate("create index idx_title on tasks(title)");
        s.close();
    }

    private void createPreparedStatements() throws SQLException {
        this.countTasks = this.conn.prepareStatement("select count(taskid) as rowcount from tasks where taskid = ?");
        this.insert = this.conn.prepareStatement("insert into tasks (taskid, tags, title) values (?, ?, ?)");
        this.update = this.conn.prepareStatement("update tasks set lastUpdated = current_timestamp, numReports = numReports + 1 where taskId = ?");
        this.updateTitle = this.conn.prepareStatement("update tasks set title = ? where taskid = ?");
        this.updateTags = this.conn.prepareStatement("update tasks set tags = ? where taskid = ?");
        this.updatedSince = this.conn.prepareStatement("select * from tasks where firstseen >= ? order by lastUpdated desc");
        this.numByTask = this.conn.prepareStatement("select numReports from tasks where taskid = ?");
        this.totalNumReports = this.conn.prepareStatement("select sum(numReports) as totalreports from tasks");
        this.totalNumTasks = this.conn.prepareStatement("select count(distinct taskid) as numtasks from tasks");
        this.lastUpdatedByTask = this.conn.prepareStatement("select lastUpdated from tasks where taskid = ?");
        this.lastTasks = this.conn.prepareStatement("select * from tasks order by lastUpdated desc");
        this.getByTag = this.conn.prepareStatement("select * from tasks where upper(tags) like upper('%'||?||'%') order by lastUpdated desc");
        this.getTags = this.conn.prepareStatement("select tags from tasks where taskid = ?");
        this.getByTitle = this.conn.prepareStatement("select * from tasks where upper(title) = upper(?) order by lastUpdated desc");
        this.getByTitleApprox = this.conn.prepareStatement("select * from tasks where upper(title) like upper('%'||?||'%') order by lastUpdated desc");
    }

    @Override
    public void sync() {
        this.fileCache.flushAll();
    }

    @Override
    public synchronized void shutdown() {
        LOG.info("Shutting down the FileTreeReportStore");
        if (this.fileCache != null) {
            this.fileCache.closeAll();
        }
        if (this.databaseInitialized) {
            block4: {
                try {
                    DriverManager.getConnection("jdbc:derby:tasks;shutdown=true");
                }
                catch (SQLException e) {
                    if (e.getSQLState().equals("08006")) break block4;
                    LOG.warn("Unable to shutdown embedded database", e);
                }
            }
            this.databaseInitialized = false;
        }
    }

    void receiveReport(String msg) {
        Matcher matcher = XTRACE_LINE.matcher(msg);
        if (matcher.find()) {
            Report r = Report.createFromString(msg);
            String xtraceLine = matcher.group(1);
            XTraceMetadata meta = XTraceMetadata.createFromString(xtraceLine);
            if (meta.getTaskId() != null) {
                TaskID task = meta.getTaskId();
                String taskId = task.toString().toUpperCase();
                BufferedWriter fout = this.fileCache.getHandle(task);
                if (fout == null) {
                    LOG.warn("Discarding a report due to internal fileCache error: " + msg);
                    return;
                }
                try {
                    fout.write(msg);
                    fout.newLine();
                    fout.newLine();
                    LOG.debug("Wrote " + msg.length() + " bytes to the stream");
                }
                catch (IOException e) {
                    LOG.warn("I/O error while writing the report", e);
                }
                try {
                    String title = null;
                    List<String> titleVals = r.get("Title");
                    if (titleVals != null && titleVals.size() > 0) {
                        title = titleVals.get(0);
                    }
                    TreeSet<String> newTags = null;
                    List<String> list = r.get("Tag");
                    if (list != null) {
                        newTags = new TreeSet<String>(list);
                    }
                    this.countTasks.setString(1, taskId);
                    ResultSet rs = this.countTasks.executeQuery();
                    rs.next();
                    if (rs.getInt("rowcount") == 0) {
                        if (title == null) {
                            title = taskId;
                        }
                        this.insert.setString(1, taskId);
                        this.insert.setString(2, this.joinWithCommas(newTags));
                        this.insert.setString(3, title);
                        this.insert.executeUpdate();
                    } else {
                        if (title != null) {
                            this.updateTitle.setString(1, title);
                            this.updateTitle.setString(2, taskId);
                            this.updateTitle.executeUpdate();
                        }
                        if (newTags != null) {
                            this.getTags.setString(1, taskId);
                            ResultSet tagsRs = this.getTags.executeQuery();
                            tagsRs.next();
                            String oldTags = tagsRs.getString("tags");
                            tagsRs.close();
                            newTags.addAll(Arrays.asList(oldTags.split(",")));
                            this.updateTags.setString(1, this.joinWithCommas(newTags));
                            this.updateTags.setString(2, taskId);
                            this.updateTags.executeUpdate();
                        }
                        this.update.setString(1, taskId);
                        this.update.executeUpdate();
                    }
                    rs.close();
                    this.conn.commit();
                }
                catch (SQLException e) {
                    LOG.warn("Unable to update metadata about task " + task.toString(), e);
                }
            } else {
                LOG.debug("Ignoring a report without an X-Trace taskID: " + msg);
            }
        }
    }

    private String joinWithCommas(Collection<String> strings) {
        if (strings == null) {
            return "";
        }
        StringBuilder sb = new StringBuilder();
        Iterator<String> it = strings.iterator();
        while (it.hasNext()) {
            sb.append(it.next());
            if (!it.hasNext()) continue;
            sb.append(",");
        }
        return sb.toString();
    }

    @Override
    public void run() {
        LOG.info("FileTreeReportStore running with datadir " + this.dataDirName);
        while (true) {
            String msg;
            if (!this.shouldOperate) {
                continue;
            }
            try {
                msg = this.incomingReports.take();
            }
            catch (InterruptedException e1) {
                continue;
            }
            this.receiveReport(msg);
        }
    }

    @Override
    public Iterator<Report> getReportsByTask(TaskID task) {
        return new FileTreeIterator(this.taskIdtoFile(task.toString()));
    }

    @Override
    public List<TaskRecord> getTasksSince(long milliSecondsSince1970, int offset, int limit) {
        ArrayList<TaskRecord> lst = new ArrayList<TaskRecord>();
        try {
            if (offset + limit + 1 < 0) {
                this.updatedSince.setMaxRows(Integer.MAX_VALUE);
            } else {
                this.updatedSince.setMaxRows(offset + limit + 1);
            }
            this.updatedSince.setString(1, new Timestamp(milliSecondsSince1970).toString());
            ResultSet rs = this.updatedSince.executeQuery();
            int i = 0;
            while (rs.next()) {
                if (i >= offset && i < offset + limit) {
                    lst.add(this.readTaskRecord(rs));
                }
                ++i;
            }
        }
        catch (SQLException e) {
            LOG.warn("Internal SQL error", e);
        }
        return lst;
    }

    @Override
    public List<TaskRecord> getLatestTasks(int offset, int limit) {
        int numToFetch = offset + limit;
        ArrayList<TaskRecord> lst = new ArrayList<TaskRecord>();
        try {
            if (offset + limit + 1 < 0) {
                this.lastTasks.setMaxRows(Integer.MAX_VALUE);
            } else {
                this.lastTasks.setMaxRows(offset + limit + 1);
            }
            ResultSet rs = this.lastTasks.executeQuery();
            int i = 0;
            while (rs.next() && numToFetch > 0) {
                if (i >= offset && i < offset + limit) {
                    lst.add(this.readTaskRecord(rs));
                }
                --numToFetch;
                ++i;
            }
            rs.close();
        }
        catch (SQLException e) {
            LOG.warn("Internal SQL error", e);
        }
        return lst;
    }

    @Override
    public List<TaskRecord> getTasksByTag(String tag, int offset, int limit) {
        ArrayList<TaskRecord> lst = new ArrayList<TaskRecord>();
        try {
            if (offset + limit + 1 < 0) {
                this.getByTag.setMaxRows(Integer.MAX_VALUE);
            } else {
                this.getByTag.setMaxRows(offset + limit + 1);
            }
            this.getByTag.setString(1, tag);
            ResultSet rs = this.getByTag.executeQuery();
            int i = 0;
            while (rs.next()) {
                TaskRecord rec = this.readTaskRecord(rs);
                if (rec.getTags().contains(tag) && i >= offset && i < offset + limit) {
                    lst.add(rec);
                }
                ++i;
            }
            rs.close();
        }
        catch (SQLException e) {
            LOG.warn("Internal SQL error", e);
        }
        return lst;
    }

    public int countByTaskId(TaskID taskId) {
        try {
            this.numByTask.setString(1, taskId.toString().toUpperCase());
            ResultSet rs = this.numByTask.executeQuery();
            if (rs.next()) {
                return rs.getInt("numreports");
            }
            rs.close();
        }
        catch (SQLException e) {
            LOG.warn("Internal SQL error", e);
        }
        return 0;
    }

    public long lastUpdatedByTaskId(TaskID taskId) {
        long ret = 0L;
        try {
            this.lastUpdatedByTask.setString(1, taskId.toString());
            ResultSet rs = this.lastUpdatedByTask.executeQuery();
            if (rs.next()) {
                Timestamp ts = rs.getTimestamp("lastUpdated");
                ret = ts.getTime();
            }
            rs.close();
        }
        catch (SQLException e) {
            LOG.warn("Internal SQL error", e);
        }
        return ret;
    }

    public List<TaskRecord> createRecordList(ResultSet rs, int offset, int limit) throws SQLException {
        ArrayList<TaskRecord> lst = new ArrayList<TaskRecord>();
        int i = 0;
        while (rs.next()) {
            if (i >= offset && i < offset + limit) {
                lst.add(this.readTaskRecord(rs));
            }
            ++i;
        }
        return lst;
    }

    @Override
    public List<TaskRecord> getTasksByTitle(String title, int offset, int limit) {
        List<TaskRecord> lst = new ArrayList<TaskRecord>();
        try {
            if (offset + limit + 1 < 0) {
                this.getByTitle.setMaxRows(Integer.MAX_VALUE);
            } else {
                this.getByTitle.setMaxRows(offset + limit + 1);
            }
            this.getByTitle.setString(1, title);
            lst = this.createRecordList(this.getByTitle.executeQuery(), offset, limit);
        }
        catch (SQLException e) {
            LOG.warn("Internal SQL error", e);
        }
        return lst;
    }

    @Override
    public List<TaskRecord> getTasksByTitleSubstring(String title, int offset, int limit) {
        List<TaskRecord> lst = new ArrayList<TaskRecord>();
        try {
            if (offset + limit + 1 < 0) {
                this.getByTitleApprox.setMaxRows(Integer.MAX_VALUE);
            } else {
                this.getByTitleApprox.setMaxRows(offset + limit + 1);
            }
            this.getByTitleApprox.setString(1, title);
            lst = this.createRecordList(this.getByTitleApprox.executeQuery(), offset, limit);
        }
        catch (SQLException e) {
            LOG.warn("Internal SQL error", e);
        }
        return lst;
    }

    @Override
    public int numReports() {
        int total = 0;
        try {
            ResultSet rs = this.totalNumReports.executeQuery();
            rs.next();
            total = rs.getInt("totalreports");
            rs.close();
        }
        catch (SQLException e) {
            LOG.warn("Internal SQL error", e);
        }
        return total;
    }

    @Override
    public int numTasks() {
        int total = 0;
        try {
            ResultSet rs = this.totalNumTasks.executeQuery();
            rs.next();
            total = rs.getInt("numtasks");
            rs.close();
        }
        catch (SQLException e) {
            LOG.warn("Internal SQL error", e);
        }
        return total;
    }

    private File taskIdtoFile(String taskId) {
        File l1 = new File(this.dataDirName, taskId.substring(0, 2));
        File l2 = new File(l1, taskId.substring(2, 4));
        File l3 = new File(l2, taskId.substring(4, 6));
        File taskFile = new File(l3, taskId + ".txt");
        return taskFile;
    }

    @Override
    public long dataAsOf() {
        return this.fileCache.lastSynched();
    }

    private TaskRecord readTaskRecord(ResultSet rs) throws SQLException {
        TaskID taskId = TaskID.createFromString(rs.getString("taskId"));
        Date firstSeen = new Date(rs.getTimestamp("firstSeen").getTime());
        Date lastUpdated = new Date(rs.getTimestamp("lastUpdated").getTime());
        String title = rs.getString("title");
        int numReports = rs.getInt("numReports");
        List<String> tags = Arrays.asList(rs.getString("tags").split(","));
        return new TaskRecord(taskId, firstSeen, lastUpdated, numReports, title, tags);
    }

    static final class FileTreeIterator
    implements Iterator<Report> {
        private BufferedReader in = null;
        private Report nextReport = null;

        FileTreeIterator(File taskfile) {
            if (taskfile.exists() && taskfile.canRead()) {
                try {
                    this.in = new BufferedReader(new FileReader(taskfile), 4096);
                }
                catch (FileNotFoundException fileNotFoundException) {
                    // empty catch block
                }
            }
            this.nextReport = this.calcNext();
        }

        @Override
        public boolean hasNext() {
            return this.nextReport != null;
        }

        @Override
        public Report next() {
            Report ret = this.nextReport;
            this.nextReport = this.calcNext();
            return ret;
        }

        private Report calcNext() {
            if (this.in == null) {
                return null;
            }
            try {
                this.in.mark(4096);
            }
            catch (IOException e) {
                LOG.warn("I/O error", e);
                return null;
            }
            String line = null;
            do {
                try {
                    line = this.in.readLine();
                }
                catch (IOException e) {
                    LOG.warn("I/O error", e);
                }
            } while (line != null && !line.startsWith("X-Trace Report ver"));
            if (line == null) {
                try {
                    this.in.reset();
                }
                catch (IOException e) {
                    LOG.warn("I/O error", e);
                }
                return null;
            }
            StringBuilder reportbuf = new StringBuilder();
            do {
                reportbuf.append(line + "\n");
                try {
                    line = this.in.readLine();
                }
                catch (IOException e) {
                    LOG.warn("I/O error", e);
                    return null;
                }
            } while (line != null && !line.equals(""));
            return Report.createFromString(reportbuf.toString());
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }

    private static final class LRUFileHandleCache {
        private File dataRootDir;
        private final int CACHE_SIZE;
        private Map<String, BufferedWriter> fCache = null;
        private long lastSynched;

        public LRUFileHandleCache(int size, File dataRootDir) throws XTraceException {
            this.CACHE_SIZE = size;
            this.lastSynched = System.currentTimeMillis();
            this.dataRootDir = dataRootDir;
            this.fCache = new LinkedHashMap<String, BufferedWriter>(this.CACHE_SIZE, 0.75f, true){

                @Override
                protected boolean removeEldestEntry(Map.Entry<String, BufferedWriter> eldest) {
                    if (this.size() > LRUFileHandleCache.this.CACHE_SIZE) {
                        BufferedWriter evicted = eldest.getValue();
                        try {
                            evicted.flush();
                            evicted.close();
                        }
                        catch (IOException e) {
                            LOG.warn("Error evicting file for task: " + eldest.getKey(), e);
                        }
                    }
                    return this.size() > LRUFileHandleCache.this.CACHE_SIZE;
                }
            };
        }

        public synchronized BufferedWriter getHandle(TaskID task) throws IllegalArgumentException {
            String taskstr = task.toString();
            LOG.debug("Getting handle for task: " + taskstr);
            if (taskstr.length() < 6) {
                throw new IllegalArgumentException("Invalid task id: " + taskstr);
            }
            if (!this.fCache.containsKey(taskstr)) {
                File l1 = new File(this.dataRootDir, taskstr.substring(0, 2));
                File l2 = new File(l1, taskstr.substring(2, 4));
                File l3 = new File(l2, taskstr.substring(4, 6));
                if (!l3.exists()) {
                    LOG.debug("Creating directory for task " + taskstr + ": " + l3.toString());
                    if (!l3.mkdirs()) {
                        LOG.warn("Error creating directory " + l3.toString());
                        return null;
                    }
                } else {
                    LOG.debug("Directory " + l3.toString() + " already exists; not creating");
                }
                File taskFile = new File(l3, taskstr + ".txt");
                try {
                    BufferedWriter writer = new BufferedWriter(new FileWriter(taskFile, true));
                    this.fCache.put(taskstr, writer);
                    LOG.debug("Inserting new BufferedWriter into the file cache for task " + taskstr);
                }
                catch (IOException e) {
                    LOG.warn("Interal I/O error", e);
                    return null;
                }
            } else {
                LOG.debug("Task " + taskstr + " was already in the cache, no need to insert");
            }
            return this.fCache.get(taskstr);
        }

        public synchronized void flushAll() {
            for (BufferedWriter writer : this.fCache.values()) {
                try {
                    writer.flush();
                }
                catch (IOException e) {
                    LOG.warn("I/O error while flushing file", e);
                }
            }
            this.lastSynched = System.currentTimeMillis();
        }

        public synchronized void closeAll() {
            this.flushAll();
            String[] taskIds = this.fCache.keySet().toArray(new String[0]);
            for (int i = 0; i < taskIds.length; ++i) {
                LOG.debug("Closing handle for file of " + taskIds[i]);
                BufferedWriter writer = this.fCache.get(taskIds[i]);
                if (writer != null) {
                    try {
                        writer.close();
                    }
                    catch (IOException e) {
                        LOG.warn("I/O error closing file for task " + taskIds[i], e);
                    }
                }
                this.fCache.remove(taskIds[i]);
            }
        }

        public long lastSynched() {
            return this.lastSynched;
        }
    }
}

