/*
 Licensed to the Apache Software Foundation (ASF) under one
 or more contributor license agreements.  See the NOTICE file
 distributed with this work for additional information
 regarding copyright ownership.  The ASF licenses this file
 to you under the Apache License, Version 2.0 (the
 "License"); you may not use this file except in compliance
 with the License.  You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing,
 software distributed under the License is distributed on an
 "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 KIND, either express or implied.  See the License for the
 specific language governing permissions and limitations
 under the License.
 */
package org.apache.devicemap;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.devicemap.loader.LoaderOption;
import org.apache.devicemap.data.Device;
import org.apache.devicemap.loader.LoaderFactory;

public class DeviceMapClient {

    private static boolean printWarnings = true;
    private static long initCount = 0;

    //indexes
    private Map<String, Device> devices;
    private Map<String, List<Device>> patterns;

    public DeviceMapClient() {
        devices = null;
        patterns = null;
    }

    public void initDeviceData(LoaderOption option) throws IOException {
        initDeviceData(option, null);
    }

    public synchronized void initDeviceData(LoaderOption option, String path) throws IOException {
        devices = LoaderFactory.getLoader(option, path).getData();

        initCount++;

        if (printWarnings && initCount % 1000 == 0) {
            Util.errorLog("Possible device data over-initialization detected");
        }

        if (devices == null) {
            patterns = null;
            return;
        }

        createIndex();
    }

    private void createIndex() {
        patterns = new HashMap<String, List<Device>>(8000);

        for (Device device : devices.values()) {
            for (List<String> patternset : device.getPatterns().getPatterns()) {
                for (int i = 0; i < patternset.size(); i++) {
                    String pattern = patternset.get(i);

                    //duplicate
                    if (patterns.get(pattern) != null) {
                        if (i == (patternset.size() - 1) && !patterns.get(pattern).contains(device)) {
                            patterns.get(pattern).add(device);
                        }
                    } else {
                        List<Device> single = new ArrayList<Device>();
                        single.add(device);
                        patterns.put(pattern, single);
                    }
                }
            }
        }
    }

    public Map<String, String> classify(String text) {
        if (devices == null) {
            throw new RuntimeException("Uninitialized device index");
        }

        Map<String, List<Device>> hits = new HashMap<String, List<Device>>(100);
        Device winner = null;
        String winnerStr = "";

        if (text == null) {
            return getUnknownDevice();
        }

        Util.debugLog("classify: '" + text + "'");

        String[] parts = text.split(" |-|_|/|\\\\|\\[|\\]|\\(|\\)|;");

        //generate ngrams upto size 4
        for (int i = 0; i < parts.length; i++) {
            String pattern = "";
            for (int j = 0; j < 4 && (j + i) < parts.length; j++) {
                if (parts[i + j].isEmpty()) {
                    continue;
                }
                
                pattern += Util.normalize(parts[i + j]);

                List<Device> dlist = patterns.get(pattern);

                if (dlist != null) {
                    hits.put(pattern, dlist);

                    for (Device device : dlist) {
                        Util.debugLog("Hit found: " + pattern + " => " + device.getId() + " " + device.getPatterns());
                    }
                }
            }
        }

        //look for the strongest hit
        for (String hit : hits.keySet()) {
            for (Device device : hits.get(hit)) {
                if (!device.getPatterns().isValid(hits.keySet())) {
                    continue;
                }

                Util.debugLog("Hit candidate: " + hit + " => " + device.getId());

                if (winner != null) {
                    if ("simple".equals(winner.getType()) && !"simple".equals(device.getType())) {
                        winner = device;
                        winnerStr = hit;
                    } else if (hit.length() > winnerStr.length() &&
                            (!"simple".equals(device.getType()) || device.getType().equals(winner.getType()))) {
                        winner = device;
                        winnerStr = hit;
                    }
                } else {
                    winner = device;
                    winnerStr = hit;
                }
            }
        }

        if (winner != null) {
            Util.debugLog("Result: " + winner);
            return winner.getAttributes();
        } else {
            return getUnknownDevice();
        }
    }

    private Map<String, String> getUnknownDevice() {
        Map<String, String> unknown = new HashMap<String, String>();
        unknown.put("id", Constants.UNKNOWN_ID);
        return unknown;
    }

    public int getDeviceCount() {
        if (devices == null) {
            return -1;
        }
        return devices.size();
    }

    public int getPatternCount() {
        if (patterns == null) {
            return -1;
        }
        return patterns.size();
    }
    
    public long getNodeCount() {
        if (patterns == null) {
            return -1;
        }
        long count=0;
        for(List<Device> devices:patterns.values()) {
            count+=devices.size();
        }
        return count;
    }

    public static void setPrintWarnings(boolean aPrintWarnings) {
        printWarnings = aPrintWarnings;
    }
}
