/*
 Licensed to the Apache Software Foundation (ASF) under one
 or more contributor license agreements.  See the NOTICE file
 distributed with this work for additional information
 regarding copyright ownership.  The ASF licenses this file
 to you under the Apache License, Version 2.0 (the
 "License"); you may not use this file except in compliance
 with the License.  You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing,
 software distributed under the License is distributed on an
 "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 KIND, either express or implied.  See the License for the
 specific language governing permissions and limitations
 under the License.
 */
package org.apache.devicemap.cmd;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.util.Map;

import org.apache.devicemap.Constants;
import org.apache.devicemap.DeviceMapClient;
import org.apache.devicemap.loader.LoaderOption;

public class Main {

    public static void main(String[] args) throws Exception {

        System.out.println("DeviceMap Java Client " + Constants.VERSION);
        
        if(args.length == 0) {
            System.out.println("Usage: -h for help");
        }

        String loaderPath = null;
        LoaderOption option = LoaderOption.UNINITIALIZED;
        String parameter = null;
        boolean debug = true;

        for (int i = 0; i < args.length; i++) {
            if (args[i].equals("-f")) {
                option = LoaderOption.FOLDER;
                if (args.length > (++i)) {
                    loaderPath = args[i];
                }
            } else if (args[i].equals("-u")) {
                option = LoaderOption.URL;
                if (args.length > (++i)) {
                    loaderPath = args[i];
                }
            } else if (args[i].equals("-j")) {
                option = LoaderOption.JAR;
            } else if (args[i].equals("-d")) {
                debug = false;
            } else if (args[i].startsWith("-h") || args[i].startsWith("--h")) {
                System.out.println("Usage: " + Main.class.getName() + " [OPTIONS] [FILE|STRING]\n");
                System.out.println("  -f <path>            load Device Map resouces from folder or \"default\"");
                System.out.println("  -j                   load Device Map resouces from jar file in classpath");
                System.out.println("  -u <url>             load Device Map resouces from URL or \"default\"");
                System.out.println("  -d                   no debug logging");
                System.out.println("  FILE                 text file of strings");
                System.out.println("  STRING               test string");

                return;
            } //[test string] | [test file]
            else if(!args[i].isEmpty()) {
                parameter = args[i];
            }
        }
        
        if(debug) {
            System.setProperty("debug", "true");
        }

        if ("default".equals(loaderPath)) {
            loaderPath = null;
        }

        DeviceMapClient client = new DeviceMapClient();

        long start = System.currentTimeMillis();

        client.initDeviceData(option, loaderPath);

        long diff = System.currentTimeMillis() - start;

        System.out.println("Loaded " + client.getDeviceCount() + " devices with " + client.getPatternCount() +
                " patterns and " + client.getNodeCount() + " nodes in " + diff + "ms");

        String test = "Mozilla/5.0 (Linux; U; Android 2.2; en; HTC Aria A6380 Build/ERE27) AppleWebKit/540.13+ (KHTML, like Gecko) Version/3.1 Mobile Safari/524.15.0";

        long startn = System.nanoTime();

        Map<String, String> m = client.classify(test);

        long diffn = (System.nanoTime() - startn) / 1000;

        System.out.println("Test lookup: '" + m.get("id") + "' time: " + diffn + "usec");

        if (parameter == null) {
        } else if ((new File(parameter)).exists()) {
            System.out.println("Text file: " + parameter);

            int count = 0;
            int total = 0;

            BufferedReader in = new BufferedReader(new FileReader(parameter));
            String line;

            while ((line = in.readLine()) != null) {
                System.out.println("Text: '" + line + "'");

                startn = System.nanoTime();

                m = client.classify(line);

                diffn = System.nanoTime() - startn;

                total += diffn;
                count++;

                System.out.println("Text lookup " + count + ": '" + m.get("id") + "' time: " + (diffn / 1000) + "usec");
            }

            in.close();

            if (count == 0) {
                count = 1;
            }

            total /= count;

            System.out.println("TOTAL lookups: " + count + ", average time: " + (total / 1000) + "usec");
        } else {
            System.out.println("Text: '" + parameter + "'");

            startn = System.nanoTime();

            m = client.classify(parameter);

            diffn = System.nanoTime() - startn;

            System.out.println("Text lookup: '" + m.get("id") + "' time: " + (diffn / 1000) + "usec");

            System.out.print("DeviceMap attributes => ");

            for (String key : m.keySet()) {
                String value = m.get(key);
                System.out.print(key + ": '" + value + "' ");
            }

            System.out.println("");
        }
    }
}
