/*
 Licensed to the Apache Software Foundation (ASF) under one
 or more contributor license agreements.  See the NOTICE file
 distributed with this work for additional information
 regarding copyright ownership.  The ASF licenses this file
 to you under the Apache License, Version 2.0 (the
 "License"); you may not use this file except in compliance
 with the License.  You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing,
 software distributed under the License is distributed on an
 "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 KIND, either express or implied.  See the License for the
 specific language governing permissions and limitations
 under the License.
 */
package org.apache.devicemap.loader.impl;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.devicemap.Constants;
import org.apache.devicemap.data.Device;
import org.apache.devicemap.Util;
import org.apache.devicemap.loader.Loader;
import org.apache.devicemap.loader.Resource;
import org.apache.devicemap.loader.parser.XMLParser;

public class DDRLoader implements Loader {

    private final Map<String, Device> devices;

    private final Resource resourceLoader;

    public DDRLoader(Resource resourceLoader) {
        devices = new HashMap<String, Device>(5000);
        this.resourceLoader = resourceLoader;
    }

    @Override
    public Map<String, Device> getData() throws IOException {
        long start = System.currentTimeMillis();

        BufferedReader ddin = new BufferedReader(new InputStreamReader(resourceLoader.getResource(Constants.DEVICE_DATA), "UTF-8"));
        loadDeviceData(ddin);
        ddin.close();

        long diff = System.currentTimeMillis() - start;
        Util.debugLog("Loaded " + Constants.DEVICE_DATA + " in " + diff + "ms");

        try {
            start = System.currentTimeMillis();

            BufferedReader ddpin = new BufferedReader(new InputStreamReader(resourceLoader.getResource(Constants.DEVICE_DATA_PATCH), "UTF-8"));
            loadDeviceData(ddpin);
            ddpin.close();

            diff = System.currentTimeMillis() - start;
            Util.debugLog("Loaded " + Constants.DEVICE_DATA_PATCH + " in " + diff + "ms");
        } catch (FileNotFoundException ex) {
            Util.debugLog("WARNING: file not found " + Constants.DEVICE_DATA_PATCH + ": " + ex.toString());
        }

        setParentAttributes();

        start = System.currentTimeMillis();

        BufferedReader bin = new BufferedReader(new InputStreamReader(resourceLoader.getResource(Constants.BUILDER_DATA), "UTF-8"));
        loadDevicePatterns(bin);
        bin.close();

        diff = System.currentTimeMillis() - start;
        Util.debugLog("Loaded " + Constants.BUILDER_DATA + " in " + diff + "ms");

        try {
            start = System.currentTimeMillis();

            BufferedReader bpin = new BufferedReader(new InputStreamReader(resourceLoader.getResource(Constants.BUILDER_DATA_PATCH), "UTF-8"));
            loadDevicePatterns(bpin);
            bpin.close();

            diff = System.currentTimeMillis() - start;
            Util.debugLog("Loaded " + Constants.BUILDER_DATA_PATCH + " in " + diff + "ms");
        } catch (FileNotFoundException ex) {
            Util.debugLog("WARNING: file not found " + Constants.BUILDER_DATA_PATCH + ": " + ex.toString());
        }

        return getDevices();
    }

    /*
     * loads device data from an InputStreamReader
     */
    private void loadDeviceData(Reader in) throws IOException {
        XMLParser parser = new XMLParser(in);
        String tag;
        Device device = new Device();
        Map<String, String> attributes = new HashMap<String, String>();

        while (!(tag = parser.getNextTag()).isEmpty()) {
            //new device found
            if (tag.startsWith("<device ")) {
                device.setId(XMLParser.getAttribute(tag, "id"));
                device.setParentId(XMLParser.getAttribute(tag, "parentId"));
            } else if (tag.equals("</device>")) {

                //add the device
                if (device.getId() != null && !device.getId().isEmpty()) {
                    attributes.put("id", device.getId());
                    device.setAttributes(attributes);
                    devices.put(device.getId(), device);
                }

                //reset the device
                device = new Device();
                attributes = new HashMap<String, String>();
            } else if (tag.startsWith("<property ")) {
                //add the property to the device
                String key = XMLParser.getAttribute(tag, "name");
                String value = XMLParser.getAttribute(tag, "value");

                attributes.put(key, value);
            }
        }
    }

    /*
     * loads patterns from an InputStreamReader
     */
    private void loadDevicePatterns(Reader in) throws IOException {
        XMLParser parser = new XMLParser(in);
        String tag;
        String builder = "";
        Device device = null;
        String id = "";
        List<String> patterns = new ArrayList<String>();

        while (!(tag = parser.getNextTag()).isEmpty()) {
            //new builder found
            if (tag.startsWith("<builder ")) {
                builder = XMLParser.getAttribute(tag, "class");

                if (builder.lastIndexOf(".") >= 0) {
                    builder = builder.substring(builder.lastIndexOf(".") + 1);
                }
            } else if (tag.startsWith("<device ")) {
                //new device found
                id = XMLParser.getAttribute(tag, "id");
                device = devices.get(id);
            } else if (tag.equals("</device>")) {
                //add the device
                if (device != null) {
                    //TwoStep is an AND pattern, also index the unigram
                    if (builder.equals("TwoStepDeviceBuilder")) {
                        device.getPatterns().setAndPattern(patterns);

                        String unigram = "";

                        for (String pattern : patterns) {
                            if (pattern.contains(unigram)) {
                                unigram = pattern;
                            } else {
                                unigram += pattern;
                            }
                        }

                        device.getPatterns().setPattern(unigram);
                    } else {
                        device.getPatterns().setOrPattern(patterns);
                    }

                    if (builder.equals("SimpleDeviceBuilder")) {
                        device.setType("simple");
                    } else {
                        device.setType("weak");
                    }
                } else {
                    Util.debugLog("ERROR: device not found: '" + id + "'");
                }

                //reset the device
                device = null;
                id = "";
                patterns = new ArrayList<String>();
            } else if (tag.equals("<value>")) {
                //add the pattern to the device
                String pattern = Util.normalize(parser.getTagValue());

                if (pattern.isEmpty()) {
                    continue;
                }

                patterns.add(pattern);
            }
        }
    }

    /**
     * Sets attributes from parents
     */
    private void setParentAttributes() {
        for (Device device : devices.values()) {
            mergeParent(device);
        }
    }

    private void mergeParent(Device device) {
        String parentId = device.getParentId();

        if (parentId == null) {
            return;
        }

        Device parent = devices.get(parentId);

        if (parent == null) {
            return;
        }

        mergeParent(parent);

        for (String key : parent.getAttributes().keySet()) {
            String value = parent.getAttributes().get(key);

            if (!device.getAttributes().containsKey(key)) {
                device.getAttributes().put(key, value);
            }
        }
    }

    /**
     * @return the devices
     */
    private Map<String, Device> getDevices() {
        return devices;
    }
}
