/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.apache.portals.applications.webcontent2.proxy.ProxyMapping;
import org.apache.portals.applications.webcontent2.proxy.ProxyMappingRegistry;


/**
 * Default simple implementation of {@link ProxyMappingRegistry}.
 */
public class DefaultProxyMappingRegistry implements ProxyMappingRegistry
{

    /**
     * Internal proxy mapping list.
     */
    private List<ProxyMapping> proxyMappings = new ArrayList<ProxyMapping>();

    /**
     * {@inheritDoc}
     */
    public boolean addProxyMapping(ProxyMapping proxyMapping)
    {
        return proxyMappings.add(proxyMapping);
    }

    /**
     * {@inheritDoc}
     */
    public boolean addAllProxyMappings(Collection<ProxyMapping> proxyMappingsCollection)
    {
        if (proxyMappingsCollection != null)
        {
            return this.proxyMappings.addAll(proxyMappingsCollection);
        }

        return false;
    }

    /**
     * {@inheritDoc}
     */
    public boolean removeProxyMapping(ProxyMapping proxyMapping)
    {
        return proxyMappings.remove(proxyMapping);
    }

    /**
     * {@inheritDoc}
     */
    public Collection<ProxyMapping> getProxyMappings()
    {
        return Collections.unmodifiableCollection(proxyMappings);
    }

    /**
     * {@inheritDoc}
     */
    public ProxyMapping findProxyMappingByLocalPath(String localPath)
    {
        if (localPath == null)
        {
            return null;
        }

        for (ProxyMapping proxyMapping : proxyMappings)
        {
            if (proxyMapping.matchesLocal(localPath))
            {
                return proxyMapping;
            }
        }

        return null;
    }

    /**
     * {@inheritDoc}
     */
    public ProxyMapping findProxyMappingByRemoteURI(URI remoteURI)
    {
        if (remoteURI == null)
        {
            return null;
        }

        for (ProxyMapping proxyMapping : proxyMappings)
        {
            if (proxyMapping.matchesRemote(remoteURI))
            {
                return proxyMapping;
            }
        }

        return null;
    }
}
