/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.zip.GZIPInputStream;

import org.apache.portals.applications.webcontent2.rewriter.Source;

/**
 * Acts as an holder for a content rewriting Source in the form of a gzipped {@link Source}.
 * <p>
 * <em>Note:</em> Due to their internal use of either a Reader or InputStream instance,
 * <code>GzippedSource</code> instances may only be used once.
 * </p>
 */
public class GzippedSource implements Source
{

    /**
     * Underlying gzipped {@link Source}.
     */
    private final Source source;

    /**
     * Character encoding to be used when {@link #getReader()} is invoked to create a
     * reader from the underlying gzipped byte stream.
     */
    private final String characterEncoding;

    /**
     * Constructs a <code>GzippedSource</code> for the underlying gzipped <code>source</code>.
     * @param source
     */
    public GzippedSource(final Source source)
    {
        this(source, null);
    }

    /**
     * Constructs a <code>GzippedSource</code> for the underlying gzipped <code>source</code>
     * with the specific <code>characterEncoding</code> which can be used when {@link #getReader()}
     * is invoked from the underlying gzipped byte stream.
     * @param source
     * @param characterEncoding
     */
    public GzippedSource(final Source source, final String characterEncoding)
    {
        this.source = source;
        this.characterEncoding = characterEncoding;
    }

    /**
     * Retrieves the gunzipped content as binary data using a {@link InputStream}.
     * <P>
     * Either this method or {@link #getReader} may be called to read the body, not both.
     * </P>
     */
    public InputStream getInputStream() throws IOException
    {
        return new GZIPInputStream(source.getInputStream());
    }

    /**
     * Retrieves the gunzipped content as character data using a {@link Reader}.
     * <P>
     * Either this method or {@link #getInputStream} may be called to read the body, not both.
     * </P>
     */
    public Reader getReader() throws IOException
    {
        if (characterEncoding != null)
        {
            return new InputStreamReader(getInputStream(), characterEncoding);
        }
        else
        {
            return new InputStreamReader(getInputStream());
        }
    }
}
